;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_RESAMPLE
;
; Purpose     :	Rebins y axis within a 2-D image using resampling. Allows pedestal shift.
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine maps the input image yaxis into a new
;               size optionally taking into account the different column to column y mapping i.e the
;               different channel sizes.
;               It also allows an option to shift each column to match pedestal positions.
;
; Use         : < ies_resample, y, lut_list, no_of_samples, ch_positions, pedestals, nchannels, yp, SHFTCH=shftch >
;
; Inputs      : y            : FLTARR(nchannels,nsamples) data values at each sample point for each channel
;               luts         : BYTARR(nsamples) LUT used for each sample point
;               ch_positions : FLTARR(2,nchannels,1,nluts) channel bin positions for each LUT.
;               pedestals    : FLTTARR(nsamples) containing pedestal positions keV for each sample
;               nchannels    : LONG no. of y channels
;               yp           : FLTARR channel or bin boundaries for each new y sample
;
; Opt. Inputs : None.
;
; Outputs     : y            : FLTARR(ysize,nsamples) rebinned image
;
; Opt. Outputs:	None.
;
; Keywords    :/SHFTCH   : indicates whether to shift data to match pedestal positions.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/11/96
;                 Adapted from ies rebin y.
;
; Modified    :	Version 0.1, MKC, 11/2/97
;                 Corrected energy channel map.
;              	Version 0.2, 2/4/97
;                 Changed yp to bin boundaries rather than bin centres.
;               Version 0.3, 2/10/97
;                 Allowed elo,ehi to be defined.
;               Version 0.4, 27/10/98
;                 Used ylo.
;               Version 0.5, 2/2/99
;                 Modified map calculation.
;               Version 0.6, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.7, 20/07/01
;                 Used ies runs.
;               Version 0.8, 24/10/03
;                 Used ies lut list.
;
; Version     :	Version 0.8, 24/10/03
;-
;**********************************************************

PRO ies_resample, y, luts, ch_positions, pedestals, nchannels, yp, SHFTCH=shftch

  ; fiddle y range of data to fit into plot box

  ; get no of data samples

  no_of_samples = N_ELEMENTS(luts)

  ; get new no. of y samples

  ysize = N_ELEMENTS(yp)-1

  ; get min and max y values

  ylo = yp(0)
  yhi = yp(ysize)

  ; get new channel centres

  yp = (yp(1:ysize)+yp(0:ysize-1))/2.0

  ; get bin or channel no.s

  yp = LONG(yp)

  ; form blank array

  yy = FLTARR(ysize,no_of_samples)

  ; get list of runs

  lut_list = ies_lut_list(luts, N_ELEMENTS(ch_positions[0,0,0,*]) )

  ; loop through runs of each lut

  FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get yp array for each lut

    ; get lut

    lut = lut_list[l].lut

    ; map is an array that gives the channel no. for each ch_ position (i.e energy [keV] or pitch angle [deg])
    ; min(ch_positions) -> max(ch_positions)
    ; change to ylo -> yhi
    ; NB dont save much if ylo > 0 but do if ylo < 0 ie pitch angle
    ; some positions will have no corresponding channel no.s
    ; ensure that map is constructed so that if integral channel present at end
    ; does not overwrite lower channels.
    ; NB expects ch_positions to be sensible integer values
    ; should modify so that map is always about 1000 elements long
    ; NB ch_position may have zero width ie lower = upper boundary

    map = REPLICATE(-1,LONG(yhi)-LONG(ylo)+1)

    FOR c = 1, nchannels DO BEGIN

      map_ps = (-LONG(ylo)+ch_positions(*,nchannels-c,0,lut))

      IF map_ps[0] LE (N_ELEMENTS(map)-1) AND map_ps[1] GT 1 THEN $
        map ( map_ps[0] > 0 : map_ps[0] > (map_ps[1]-1) < (N_ELEMENTS(map)-1) ) = nchannels-c

    ENDFOR

    ; translate linear map into channel map

    ypp = map ( yp - LONG(ylo) )

    list = WHERE(ypp GE 0, count)

    IF count GT 0 THEN yy(list,lut_list[l].st:lut_list[l].en) = y(ypp(list),lut_list[l].st:lut_list[l].en)

  ENDFOR

  y = yy

  ; shift bins
  ; NB perform after y expansion

  IF KEYWORD_SET(shftch) THEN BEGIN

    FOR k = 0, no_of_samples-1 DO BEGIN

      ; get copy of energies for sample

      temp = y(*,k)

      ; shift column to line pedestals along 0

      pos = LONG ( pedestals(k) * FLOAT(ysize) / (yhi-ylo) )

      IF pos GT 0 THEN BEGIN

        ; shift down

        y(0:ysize-pos-1,k) = temp(pos:*)
        y(ysize-pos:*,k)   = 0

      ENDIF ELSE IF pos LT 0 THEN BEGIN

        ; shift up

        y(-pos:*,k) = temp(0:ysize+pos-1)
        y(0:-pos-1,k) = 0

      ENDIF

    ENDFOR

  ENDIF

END