;+
; Project     :	CLUSTER - RAPID - PULSAUR
;
; Name        :	IES_RESTORE_FILE
;
; Purpose     :	Restores IF file and associated files.
;
; Explanation : Used within IES_GETSET to restore IF file.
;
; Use         : <st = ies_getset ( nbands, input_files, GROUP=group) >
;
; Inputs      : nbands : INT no. of bands to be selected from input data
;                        If nbands equals nchannels then bands are set to channels
;                        i.e as for spectral plot.
;
; Opt. Inputs : None.
;
; Outputs     : st : structure containing data selection information.
;                     lb        : INTARR(nbands) : Lower channel bound on each band.
;                     ub        : INTARR(nbands) : Upper channel bound on each band.
;                     use_bands : INTARR(nbands) : Indicates whether each band to be used.
;                     sttime    : start time of period of interest in internal format
;                     entime    : end time of period of interest in internal format
;                     use_dets  : INTARR(ndetectors) indicating whether each detector to be used
;
; Opt. Outputs:	None.
;
; Keywords    : GROUP : Widget ID of parent.
;
; Written     :	Version 0.0, Martin Carter, RAL, 5/6/95
;
; Modified    :	Version 0.1, 08/05/02
;                 Allowed use from ies_plot.
;                 Changed to function.
;                 Adapted to include CLUSTER ephem, mf and at data.
;               Version 0.2, 12/07/02
;                 Used ies_tag_exists.
;               Version 0.3, 27/02/03
;                 Avoided reading RAPID AT file twice.
;               Version 0.4, 29/09/03
;                 Changed ies_do_format argument list.
;                 Used new routine for instrument test.
;                 Calculated mf from mvals if necessary and possible.
;
; Version     :	Version 0.4, 29/09/03
;-
;**********************************************************

FUNCTION ies_restore_file, path, ADD=add, NEW=new

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  COMMON ies_getset_block, struct

  ; check if info window valid

  id_ok = 0
  IF N_ELEMENTS(struct) GT 0 THEN IF WIDGET_INFO(struct.id_infoid, /VALID_ID) THEN id_ok=1

  ; choose restore file

  restore_file = ies_pickfile ( PATH=path, FILTER='*.*', /MUST_EXIST, /READ, TITLE='Select input data file' )

  IF restore_file EQ '' THEN BEGIN
    IF id_ok THEN BEGIN
      RETURN, 0
    ENDIF ELSE BEGIN
      MESSAGE, 'Data file not selected',/INFORMATIONAL
      RETURN, 0
    ENDELSE
  ENDIF

  ; check if compressed file

  IF ies_test_gzip(restore_file) THEN BEGIN
    IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Decompressing data file '+restore_file
    ies_gunzip, restore_file
  ENDIF

  ; restore data file
  ; assume directory name already added
  ; so can select directory in pickfile

  ; check whether file already restored

  IF KEYWORD_SET(add) THEN BEGIN

    dummy = WHERE ( files EQ restore_file, count )

    IF  count GT 0 THEN BEGIN
      xack, 'Specified file already restored', /MODAL
      RETURN, 0
    ENDIF

  ENDIF

  ; print some info

  IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Restoring file : ' + strip_dirname(restore_file)
  IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE=STRING(10B)+'Please wait', /APPEND

  ; set up identifiers

  input_header = 0
  input_data   = 0

  RESTORE, restore_file

  IF ies_instrument(input_header.datatype,/POLAR) OR ies_instrument(input_header.datatype,/RAPID) THEN BEGIN

    ; get ephemeris data for midpoint of range

    new_ephem = get_sc_ephem ( input_header )

    ; get attitude data for midpoint of range

    new_at = get_sc_at ( input_header )

    ; get magnetic field data for midpoint of range

    new_mf = get_sc_mf ( input_header, input_data, AT=new_at )

  ENDIF ELSE BEGIN

    new_ephem = 0

    new_at = 0

    new_mf = 0

  ENDELSE

  IF KEYWORD_SET ( new) THEN BEGIN

    ; save file name

    files = [ '', restore_file]

    ; set up full_header and full_data

    full_header = input_header
    full_data   = input_data

    ; set up ephem, at and mf

    ephem = new_ephem

    at = new_at

    mf = new_mf

  ENDIF ELSE BEGIN

    ; combine data

    ies_concatenate, full_data, input_data, HEADER=full_header, ERROR=error

    IF KEYWORD_SET(error) THEN BEGIN

      IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Incompatible data sets'

      RETURN, 0

    ENDIF

    ; save file name

    files = [files, restore_file]

    ; set up ephem, at and mf

    IF NOT KEYWORD_SET(ephem) THEN ephem = new_ephem $
    ELSE IF KEYWORD_SET(new_ephem) THEN BEGIN
      ies_concatenate, ephem, new_ephem, ERROR=error
      IF KEYWORD_SET(error) THEN IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Incompatible ephem data'
    ENDIF

    IF NOT KEYWORD_SET(at) THEN at = new_at $
    ELSE IF KEYWORD_SET(new_at) THEN BEGIN
      ies_concatenate, at, new_at, ERROR=error
      IF KEYWORD_SET(error) THEN IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Incompatible at data'
    ENDIF

    IF NOT KEYWORD_SET(mf) THEN mf = new_mf $
    ELSE IF KEYWORD_SET(new_mf) THEN BEGIN
      ies_concatenate, mf, new_mf, ERROR=error
      IF KEYWORD_SET(error) THEN IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Incompatible mf data'
    ENDIF

  ENDELSE

  ; if first time then return

  IF NOT KEYWORD_SET(struct) THEN RETURN, 1

  ; make sttime invalid to force initial start and stop times to limits

  struct.sttime     = full_header.tstart - 100000.0d0

  ; set up new common block

  blok_ies_getset, full_header.ndetectors, full_header.nsectors, full_header.tstart, full_header.tstop

  ; set new bands values
  ; if clean data then set nbands=-1
  ; NB check whether clean tag present or not first

  IF ies_tag_exists(full_header,'CLEAN') THEN clean = full_header.clean ELSE clean = 0

  ; check if integral channel present

  IF clean EQ 0 AND (NOT ies_instrument(full_header.datatype,/HIST)) AND (NOT ies_instrument(full_header.datatype,/RAPID)) THEN def_bands = 0 ELSE def_bands = -1

  bands=ies_bands(SET_CURRENT=[full_header.nbands, def_bands])

  IF bands.nbands EQ 0 THEN dummy = '   BOTTOM  CHANNEL : ' + STRTRIM(bands.botchan,2) + $
                                    '   TOP CHANNEL : ' + STRTRIM(bands.topchan,2)  $
                       ELSE dummy = '   NUMBER OF ENERGY BANDS : ' + STRTRIM(bands.nbands,2)

  ; get no. of bands

  band_list = WHERE ( bands.use_bands, nbands )

  ;IF bands.nbands EQ 0 THEN end_band = bands.topchan ELSE end_band = nbands-1

  IF nbands LE 0 THEN BEGIN
    IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='WARNING, No energy bands selected'
  ENDIF ELSE BEGIN
     dummy = dummy + '   INTEGRAL CHANNEL : '+$
                STRTRIM(bands.lb(band_list(0)),2)+'-'+STRTRIM(bands.ub(band_list(0)),2)
;                STRTRIM(bands.lb(band_list(end_band)),2)+'-'+STRTRIM(bands.ub(band_list(end_band)),2)
  ENDELSE

  WIDGET_CONTROL, struct.id_energy, SET_VALUE=dummy

  ; stop updates to screen
  ; this prevents widget jiggling around, saves time and most importantly
  ; stop extra space being created when destroy and recreate child widgets

  WIDGET_CONTROL, struct.id_top, UPDATE=0

  ; destroy old child widgets

  WIDGET_CONTROL, struct.id_files_child, /DESTROY
  WIDGET_CONTROL, struct.id_times_child, /DESTROY
  WIDGET_CONTROL, struct.id_detectors_child, /DESTROY
  WIDGET_CONTROL, struct.id_format_child, /DESTROY

  ; set up files widget

  ies_do_files, struct, files

  ; set up times widget

  ies_do_times, struct, full_header

  ; set up detectors widget

  ies_do_detectors, struct, full_header

  ; set CAL, DIV and RAT

  output_info.cal = full_header.cal
  output_info.div = full_header.div
  output_info.rat = full_header.rat

  ; set format menu

  output_info.format = ies_do_format( struct, mf, output_info.format)

  ; print info

  IF id_ok THEN WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Ready for input'

  ; set up values
  ; do this after ies_do_format because may change procname

  ies_set_values, full_header, struct, files, output_info.format, output_info.procname, ephem, at, mf

  ; update widgets

  WIDGET_CONTROL, struct.id_top, UPDATE=1

  ; return successful

  RETURN, 1

END
