;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_SHIFT
;
; Purpose     : Corrects pedestal offset and resamples data.
;
; Explanation :	Applies pedestal shift and rebins data.
;
; Use         : < ies_shift, lut_list, nsectors, ndetectors, nbands, ch_positions, new_positions, shifted_pedestals, output_array>
;
; Inputs      : lut_list      : STRUCTURE ARRAY containing LUT run info.
;               nsectors      : INT no. of sectors.
;               nenergies     : INT new no. of energies.
;               ndetectors    : INT no. of detectors.
;               nbands        : INT no. of energy channels.
;               ch_positions  : INTARR(2,nbands,ndetectors,nluts) containing original energy channel positions in bins for each detector and LUT
;               new_positions : INTARR(2,nenergies,ndetectors,nluts) containing new energy channel positions in bins for each detector and LUT
;               shifted_pedestals : FLTARR ( nsamples, nsectors, ndetectors) pedestal bin positions
;               output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nenergies, npoints) rebinned data.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	ies_div, ies_interp.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 1.0, Martin Carter, RAL, 26/9/96
;                 Adapted from ies_calibrate.
;
; Modified    :	Version 2.0, 11/11/96
;                 New ies rebin y.
;               Version 2.1, 16/1/97
;                 Renamed from ies_clean.
;                 Changed into a procedure.
;                 Set up clean version no. elsewhere.
;                 Updated nbands.
;                 Modified clean bin boundaries.
;                 Changed minimum used to minimum non-zero value rather than 0.1*mnzv
;               Version 2.2, 2/4/97
;                 Replaced ies rebin y with ies_interp and used channel boundaries rather
;                 than channel centres.
;                 Corrected bug : previously would have only used sector 0 pedestal shifts for all sectors.
;                 Changed new bin start to bin 6 rather than bin 5.
;               Version 2.3, 4/3/98
;                 Changed so that ies_interp uses undivided data.
;                 Changed so that uses bins rather than energies.
;                 Moved divide outside routine.
;                 Changed so that does not update nbands.
;                 Changed new bin boundaries.
;               Version 2.4, 3/2/99
;                 Do not pass original integral channel to ies_interp.
;                 Do not pass new integral channel boundary to ies_interp.
;                 Added CUMULATIVE switch to ies_interp.
;               Version 2.5, 7/9/00
;                 Removed restriction on no. of energy channels.
;               Version 2.6, 10/10/00
;                 Incorporated RAPID resampling.
;               Version 2.7, 18/07/01
;                 Changed RAPID flag.
;               Version 2.8, 15/08/01
;                 Adjusted shifted energies for RAPID.
;                 Previously :
;                    new_bin_start=10
;                    new_bin_sizes = [11,8,12,15,22,31,42,59]
;                 Added FLAGS keyword.
;               Version 2.9, 24/10/03
;                 Changed to lut_list argument.
;               Version 3.0, 17/12/03
;                 Added new_positions argument.
;                 Removed RAPID keyword.
;
; Version     :	Version 3.0, 17/12/03
;-
;**********************************************************

PRO ies_shift, lut_list, nsectors, nenergies, ndetectors, nbands, ch_positions, new_positions, shifted_pedestals, output_array, FLAGS=flags

  npoints = lut_list[N_ELEMENTS(lut_list)-1].en+1

  ; get data min ignoring zeros
  ; can ignore flags since these will not decrease data if set to 255 and are zeroed otherwise

  non_zero_list = WHERE ( output_array GT 0, count)

  IF count EQ 0 THEN minimum = 1 ELSE minimum = MIN(output_array(non_zero_list))

  ; last channel is integral channel so does not need end boundary

  new_yp = FLOAT(REFORM(new_positions[0,*,0,0]))

  ; get array for new output data

  new_output = FLTARR ( nsectors, ndetectors, nenergies, npoints)

  ; pedestal shift and rebin for each sector/detector

  FOR s = 0, nsectors-1 DO FOR d = 0, ndetectors-1 DO BEGIN

    ; get temp array for output

    temp = REFORM ( output_array(s,d,*,*), nbands, npoints )

    ; rebin y data

    ies_interp, temp, lut_list, ch_positions(*,*,d,*), nbands<15, new_yp, minimum, shifted_pedestals(*,s,d), $
                /SHFTCH, /CUMULATIVE, FLAGS=flags

    ; set output array

    new_output(s,d,*,*) = temp

  ENDFOR

  ; set up output

  output_array = new_output

END
