;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_SINPLOT
;
; Purpose     :	Does multiple lineplot in single graphics window.
;
; Explanation :	Does not assume that data is evenly sampled in time.
;               Common parameters to ies_sp_plot.
;
; Use         : < ies_sinplot, x, e, y, luts, flags, ch_positions, minimum, maximum, count_cal, $
;                              npoints, nchannels, minbin, maxbin, tstart, tend, $
;                              TITLE=title, XTITLE=xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
;                              BAND_LABELS=band_labels, $
;                              BUTTONS=buttons, SAVE_CC=save_cc, $
;                              NO_XAXIS=no_xaxis, NO_ERASE=no_erase, NO_MODES=no_modes, $
;                              XMARGIN=xmargin, YMARGIN=ymargin, LOG=log, SHRINK=shrink >
;
; Inputs      : x            : DBLARR(npoints) times for each sample in TAI format
;               e            : DBLARR(npoints) end times for each sample in TAI format
;               y            : FLTARR(nchannels,npoints) data values at each sample point for each channel
;               luts         : BYTARR(npoints) LUT used for each sample point
;               flag         : BYTARR(npoints) data quality flag for each sample point
;               ch_positions : BYTARR(2,nchannels,1,nluts) channel bin positions for each LUT.
;               shifted_pedestals : FLTARR(nsamples) pedestal position in kev
;               minimum      : FLOAT min. axis value
;               maximum      : FLOAT max. axis value
;               count_cal    : FLTARR(nluts) calibration factor for each LUT for counts to flux.
;               npoints      : LONG no. of sample points
;               nchannels    : LONG no. of energy channels
;               ylo          : FLOAT min. y axis value
;               yhi          : FLOAT max. y axis value
;               tstart       : DOUBLE start time in seconds for display
;               tend         : DOUBLE end time in seconds for display
;
; Opt. Inputs : None.
;
; Outputs     : Draws plot on current graphics device.
;
; Opt. Outputs:	None.
;
; Keywords    : TITLE       : STRING Title of plot.
;               XTITLE      : STRING time axis title.
;               YTITLE      : STRING energy axis title.
;               ZTITLE      : STRING counts axis title.
;               BAND_LABELS : STRARR(nchannels) Label for each energy band.
;               BUTTONS     : INTARR flags for various actions :
;                              0 : indicates whether to draw lines between data.
;                              1 : indicates whether to draw points for data.
;               SAVE_CC     : INT flag indicating that should save coordinate conversion to named variable.
;               NO_XAXIS    : INT flag preventing x axis display.
;               NO_ERASE    : INT flag preventing erase of plot device initially.
;               NO_MODES    : INT flag preventing display of flags/LUT info.
;               XMARGIN     : INTARR(2) character size of left and right margins.
;               YMARGIN     : INTARR(2) character size of top and bottom margins.
;               LOG         : INT flag indicating logarithmic display required.
;               SHRINK      : INT flag indicating whether to allow x axis space around plotted data
;
; Calls       :	ies_do_axes, ies_do_flags, ies_do_labels.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, MKC, 17/1/96
;                 Changed format of y input array.
;               Version 0.2, MKC, 22/1/96
;                 Added title keyword.
;               Version 0.3, MKC, 23/9/96
;                 Moved definition of ztitle.
;                 Moved plot title.
;               Version 0.4, 8/11/96
;                 Added PROFS KEYWORD.
;                 Added time_start and time_end variables.
;               Version 0.5, 11/11/96
;                 Added pedestals.
;                 Removed cal keyword.
;               Version 0.6, 21/1/97
;                 Removed PROFS keyword.
;                 Added no bad data plot.
;               Version 0.7, 2/4/97
;                 Added minbin to argument list.
;               Version 0.8, 12/5/97
;                 Changed time_start and time_end to be input times rather than data times
;                 so that can manipulate time display.
;               Version 0.9, 29/8/97
;                 Changed minbin, maxbin to ylo, yhi.
;               Version 1.0, 9/10/98
;                 Added ncolors to argument list.
;               Version 1.1, 12/11/98
;                 Changed button assignements.
;               Version 1.2, 5/10/00
;                 Generalized bad data flags : '7e'x --> '70'x
;               Version 1.3, 14/2/01
;                 Added e argument to procedure call.
;               Version 1.4, 5/3/1
;                 Changed x-tstart to x-time_start.
;               Version 1.5, 07/02/02
;                 Used separate function for checking good data.
;               Version 1.6, 14/05/02
;                 Changed so that uses PSYm=points rather than PSYM=1 for plotting points.
;               Version 1.7, 20/06/02
;                 Ensured only uses valid plotting symbols.
;               Version 1.8, 09/07/02
;                 Removed ncolors from argument lists.
;
; Version     :	Version 1.8, 09/07/02
;-
;**********************************************************

PRO ies_sinplot, x, e, y, luts, flags, ch_positions, minimum, maximum, count_cal, npoints, nchannels, ylo, yhi, tstart, tend, $
                 TITLE=title, XTITLE=xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
                 BAND_LABELS=band_labels, $
                 BUTTONS=buttons, SAVE_CC=save_cc, $
                 NO_XAXIS=no_xaxis, NO_ERASE=no_erase, NO_MODES=no_modes, $
                 XMARGIN=xmargin, YMARGIN=ymargin, LOG=log, SHRINK=shrink

  ; check buttons

  IF KEYWORD_SET(buttons) THEN BEGIN
    points = buttons(1)
    nobad  = buttons(2)
    lines  = buttons(4)
  ENDIF ELSE BEGIN
    lines  = 0
    points = 0
    nobad  = 0
  ENDELSE

  ; set up defaults

  IF NOT KEYWORD_SET(xmargin) THEN xmargin = [10,2]
  IF NOT KEYWORD_SET(ymargin) THEN ymargin = [4,2]

  ; set up time_start and time_end

  time_start = tstart
  time_end   = tend

  ; set up axes

  ies_do_axes, time_start, time_end, minimum, maximum, xmargin, ymargin, $
               XTITLE=xtitle, YTITLE=ztitle, SAVE_CC=save_cc, NO_XAXIS=no_xaxis, NO_ERASE=no_erase, LOG=log, SHRINK=shrink

  ; remove offset in internal time

  x = x - time_start

  ; fix data within min and max

  y = maximum < y > minimum

  ; get good data list

  IF nobad THEN BEGIN

     good_list = ies_good_data(flags, COUNT=count)

     IF count LT 2 THEN BEGIN
       MESSAGE, 'INSUFFICIENT DATA FOR PLOT', /CONT, /TRACEBACK
       RETURN
     ENDIF

     y = y(*,good_list)
     x = x(good_list)
     luts = luts(good_list)
     flags = flags(good_list)

  ENDIF

  ; check if sufficient colours

  IF nchannels LT ies_ncolors()/2 THEN BEGIN

    ; loop over overlays

    FOR k = 0, nchannels-1 DO BEGIN

      ; get color for overlay

      col = ies_get_color(k)

      ; plot y for overlay

      IF KEYWORD_SET(lines)  THEN OPLOT, x, y( k, * ), COLOR=col
      IF KEYWORD_SET(points) THEN OPLOT, x, y( k, * ), COLOR=col, PSYM=points < 7

    ENDFOR

  ENDIF ELSE BEGIN

    ; loop over overlays

    FOR k = 0, nchannels-1 DO BEGIN

      ; plot y for overlay

      IF KEYWORD_SET(lines)  THEN OPLOT, x, y( k, * ), LINESTYLE=k
      IF KEYWORD_SET(points) THEN OPLOT, x, y( k, * ), LINESTYLE=k, PSYM=(1+k) < 7

    ENDFOR

  ENDELSE

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ; plot flags and luts

  IF NOT KEYWORD_SET(no_modes) THEN ies_do_flags, x, luts, flags

  ; put down labels
  ; NB the different energy bands correspond to the ytitle

  IF KEYWORD_SET(band_labels) THEN ies_do_labels, nchannels, band_labels, $
                                                  TITLE=title, CTITLE='Bands', NO_XAXIS=no_xaxis

END
