;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_SP_PLOT
;
; Purpose     :	Does single spectral plot of counts per channel vs time.
;
; Explanation :	Produces colour coded plot of counts in channels against time.
;               If no channel sizes specified then one is assumed.
;               Does not assume that data is evenly sampled in time.
;               Allows choice of polygon plot or 2-D image method.
;               Common parameters to ies_sin_plot.
;
; Use         : < ies_sp_plot, x, e, y, luts, flags, ch_positions, minimum, maximum, count_cal,  $
;                              npoints, nchannels, ylo, yhi, tstart, tend,$
;                              TITLE=title, XTITLE=xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
;                              BAND_LABELS=band_labels, $
;                              BUTTONS=buttons, SAVE_CC=save_cc, $
;                              NO_XAXIS=no_xaxis, NO_ERASE=no_erase, NO_MODES=no_modes, $
;                              XMARGIN=xmargin, YMARGIN=ymargin, LOG=log, SHRINK=shrink >
;
; Inputs      : x            : DBLARR(npoints) times for each sample in internal format
;               e            : DBLARR(npoints) end times for each sample in TAI format if present
;               y            : FLTARR(nchannels,npoints) data values at each sample point for each channel
;               luts         : BYTARR(npoints) LUT used for each sample point
;               flag         : BYTARR(npoints) data quality flag for each sample point
;               ch_positions : FLTARR(2,nchannels,1,nluts) channel bin positions for each LUT.
;               minimum      : FLOAT min. data value
;               maximum      : FLOAT max. data value
;               count_cal    : FLTARR(nluts) calibration factor for each LUT for counts to flux.
;               npoints      : LONG no. of sample points
;               nchannels    : LONG no. of y channels
;               ylo          : FLOAT min. y axis value
;               yhi          : FLOAT max. y axis value
;               tstart       : DOUBLE start time in seconds for display
;               tend         : DOUBLE end time in seconds for display
;
; Opt. Inputs : None.
;
; Outputs     : Draws plot on current graphics device.
;
; Opt. Outputs:	None.
;
; Keywords    : TITLE       : STRING Title of plot.
;               XTITLE      : STRING time axis title.
;               YTITLE      : STRING energy axis title.
;               ZTITLE      : STRING counts axis title.
;               BAND_LABELS : STRARR(nchannels) Label for each energy band.
;               BUTTONS     : INTARR flags for various actions.
;               SAVE_CC     : INT flag indicating that should save coordinate conversion to named variable.
;               NO_XAXIS    : INT flag preventing x axis display.
;               NO_ERASE    : INT flag preventing erase of plot device initially.
;               NO_MODES    : INT flag preventing display of flags/LUT info.
;               XMARGIN     : INTARR(2) character size of left and right margins.
;               YMARGIN     : INTARR(2) character size of top and bottom margins.
;               LOG         : INT flag indicating logarithmic display required.
;               SHRINK      : INT flag indicating whether to allow x axis space around plotted data
;               PROFS       : Signals profiles plot required.
;
; Calls       :	ies_do_axes, ies_mk_image, ies_do_flags, ies_do_colorbar.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, MKC, 24/8/95
;               Version 0.2, MKC, 17/1/96
;                 Changed format of y input array.
;               Version 0.3, MKC, 19/9/96
;                 Moved units adjustment for showch.
;                 Added DIV keyword to ies_mk_image call.
;                 Changed bins to keV.
;                 Moved definition of ztitle.
;                 Added interpolate button.
;                 Fix data within min. and max internally.
;                 Removed centroid calculation.
;                 Moved plot title.
;               Version 0.4, 8/11/96
;                 Added PROFS KEYWORD.
;                 Added time_start and time_end variables.
;               Version 0.5, 11/11/96
;                 New ies_mk_image argument list.
;                 Removed cal keyword.
;               Version 0.6, 29/11/96
;                 Made 0 counts = white in postscript output.
;               Version 0.7, 21/1/97
;                 Removed PROFS KEYWORD.
;                 Added no bad data plot.
;               Version 0.8, 1/4/97
;                 Added E log and minbin.
;               Version 0.9, 12/5/97
;                 Changed time_start and time_end to be input times rather than data times
;                 so that can manipulate time display.
;               Version 1.0, 2/3/98
;                 Removed interpolation option. Removed minimum argument from ies_mk_image.
;               Version 1.1, 9/10/98
;                 Added ncolors to argument list.
;               Version 1.2, 5/10/00
;                 Generalized bad data flags : '7e'x --> '70'x
;               Version 1.3, 14/2/01
;                 Added e argument to procedure call.
;               Version 1.4, 5/3/1
;                 Changed x-tstart to x-time_start.
;               Version 1.5, 14/3/01
;                 Adjusted size of image and position of axes.
;               Version 1.6, 07/02/02
;                 Used separate function for checking good data.
;               Version 1.7, 09/07/02
;                 Removed ncolors from argument lists.
;
; Version     :	Version 1.7, 09/07/02
;-
;**********************************************************

PRO ies_sp_plot, x, e, y, luts, flags, ch_positions, minimum, maximum, count_cal, npoints, nchannels, ylo, yhi, tstart, tend, $
                 TITLE=title, XTITLE=xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
                 BAND_LABELS=band_labels, $
                 BUTTONS=buttons, SAVE_CC=save_cc, $
                 NO_XAXIS=no_xaxis, NO_ERASE=no_erase, NO_MODES=no_modes, $
                 XMARGIN=xmargin, YMARGIN=ymargin, LOG=log, SHRINK=shrink, EPHEM=ephem

  ; check buttons

  IF KEYWORD_SET(buttons) THEN BEGIN
    showch = buttons(0)
    nobad  = buttons(2)
    elog   = buttons(3)
  ENDIF ELSE BEGIN
    showch = 0
    nobad  = 0
    elog   = 0
  ENDELSE

  ; set up defaults

  IF NOT KEYWORD_SET(xmargin) THEN xmargin = [10,2]
  IF NOT KEYWORD_SET(ymargin) THEN ymargin = [4,2]

  ; set up time_start and time_end

  time_start = tstart
  time_end   = tend

  ; assume x, y, ch_positions passed by value so can alter without fear
  ; mangle x and y as desired

  ; set up axes

  ies_do_axes, time_start, time_end, ylo, yhi, xmargin, ymargin, $
    XTITLE=xtitle, YTITLE=ytitle, SAVE_CC=save_cc, NO_XAXIS=no_xaxis, NO_ERASE=no_erase, SHRINK=shrink, LOG=elog

  ; remove offset in internal time

  x = x - time_start

  e = e - time_start

  ; get good data list

  IF nobad THEN BEGIN

     good_list = ies_good_data(flags, COUNT=count)

     IF count LT 2 THEN BEGIN
       MESSAGE, 'INSUFFICIENT DATA FOR PLOT', /CONT, /TRACEBACK
       RETURN
     ENDIF

     y = y(*,good_list)
     x = x(good_list)
     e = e [good_list]
     luts = luts(good_list)
     flags = flags(good_list)

  ENDIF

  ; plot channels using TV

  ; draw image in plot box, check if rescaleable output device

  IF !D.FLAGS AND 1 THEN BEGIN

        ; fix array at some nominal size

        xsize = FIX ( N_ELEMENTS(x)*1.1 ) < 512
        ysize = yhi < 128

        ; resize array
        ; NB updates y

        ies_mk_image, y, x, luts, ch_positions, nchannels, ylo, yhi, xsize, ysize, $
                      SHOWCH=showch, SHFTCH=shftch, LOG=elog, ENDTIME=e

        ; colour scale

        ies_color_scale,{minimum:minimum,maximum:maximum,log:log},y

        ; white out zero elements for postscript plot

        IF !D.NAME EQ 'PS' THEN BEGIN

           ; list = WHERE ( y EQ 0, count )

           ; IF count GT 0 THEN y(list) = 255

        ENDIF

        ; plot array
        ; can use data coords but not as direct

        TV, y, !X.WINDOW(0), !Y.WINDOW(0), XSIZE=!X.WINDOW(1)-!X.WINDOW(0), YSIZE=!Y.WINDOW(1)-!Y.WINDOW(0), /NORMAL, ORDER=0

 ENDIF ELSE BEGIN

        ; get device coords of plot window

        vector = CONVERT_COORD ( !X.WINDOW, !Y.WINDOW, /NORMAL, /TO_DEVICE )

        ; get origin and size in device co-ords (inclusive of axes)

        xorig = vector(0,0)
        yorig = vector(1,0)

        xsize = FIX(vector(0,1) - xorig + 1)
        ysize = FIX(vector(1,1) - yorig + 1)

        ; removed + 1 because was getting overflow on widget plots 29/10/98
        ;xsize = FIX(vector(0,1) - xorig)
        ;ysize = FIX(vector(1,1) - yorig)

        ; check xsize and ysize are reasonable

        IF xsize LE 0 OR ysize LE 0 THEN BEGIN
          PRINT, 'IMAGE WINDOW INVALID SIZE : ', title
          RETURN
        ENDIF

        ; resize array
        ; NB updates y

        ies_mk_image, y, x, luts, ch_positions, nchannels, ylo, yhi, xsize, ysize, $
                      SHOWCH=showch, SHFTCH=shftch, LOG=elog, ENDTIME=e

        ; colour scale

        ies_color_scale,{minimum:minimum,maximum:maximum,log:log},y

        ; plot array
        ; added y=+1 to avoid image appearing underneath axes

        TV, y, xorig, yorig+1, ORDER=0

  ENDELSE

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ; plot flags and luts

  IF NOT KEYWORD_SET(no_modes) THEN ies_do_flags, x, luts, flags

  ; draw color bar

  ies_do_colorbar, minimum, maximum, TITLE=title, CTITLE=ztitle, NO_XAXIS=no_xaxis, LOG=log

END
