;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_TEXT_PLOT
;
; Purpose     :	Plot data as text file.
;
; Explanation : Plots data from a number of detectors as text file.
;
; Use         : < ies_text_plot >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : Plots data to file.
;
; Opt. Outputs:	None.
;
; Keywords    : /POLAR      = FLAG use POLAR output area
;               /RAPID      = FLAG use RAPID output area

;
; Calls       :	strip_dirname, anytim2cal, concat_dir, ies_ctitle, ies_pickfile.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    : Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 16/9/96
;
; Modified    : Version 0.1, 7/2/97
;                 Added more details in printout.
;               Version 0.2, 15/4/97
;                 Changed so that DIV, CAL and RAT may be appliee to clean data.
;               Version 0.3, 30/9/98
;                 Corrected bug on output.
;               Version 0.4, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;               Version 0.5, 10/11/98
;                 Added detector average.
;               Version 0.6, 13/12/99
;                 Removed SET_PLOT on error.
;                 Replaced PICKFILE with ies_pickfile
;               Version 0.7, 15/05/02
;                 Changed wording of output.
;               Version 0.8, 30/09/03
;                 Added RAPID/POLAR keyword switch.
;                 Used ies get dir.
;
; Version     :	Version 0.8, 30/09/03
;-
;**********************************************************

PRO ies_text_plot, POLAR=polar, RAPID=rapid

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  ON_IOERROR, IOERROR

  ; print some information

  WIDGET_CONTROL, ids.infoid, SET_VALUE='Printing data, please wait'

  path = ies_get_dir('TXT',POLAR=polar,RAPID=rapid)

  file = ies_pickfile (TITLE='Select file for TEXT output', FILTER=['*.txt'], PATH=path )

  IF file EQ '' THEN GOTO, ENDING

  ; open file

  OPENW, unit, file, /GET_LUN

  ; do heading

  ; get today

  get_utc, utc

  ; get list of unique luts

  luts = data(SORT(data.lut)).lut

  luts = FIX(luts(UNIQ(luts))) ; NB convert to INT else STRING conversion misbehaves

  ; define z title

  ztitle = ies_ctitle(header,output_info)

  IF (output_info.procname EQ 'SPECTRAL' OR output_info.procname EQ 'LINEPLOT') $
    AND (output_info.format EQ 'ENERGY vs DETECTOR') THEN BEGIN

    PRINTF, unit, 'ENERGY vs DETECTOR DATA for ' + anytim2cal(header.tstart, FORM=9) + ' to ' + $
                                              anytim2cal(header.tstop, FORM=9)
    PRINTF, unit, 'COMPILED ' + anytim2cal(utc, FORM=9)
    PRINTF, unit, 'NDETECTORS = ', STRTRIM(header.ndetectors,1)
    PRINTF, unit, 'NBANDS     = ', STRTRIM(output_info.ysamples,1)
    PRINTF, unit, 'NPOINTS    = ', STRTRIM(header.npoints,1)
    IF output_info.cal OR header.cal NE 0 THEN PRINTF, unit, 'CALIBRATED DATA' ELSE PRINTF, unit, 'UNCALIBRATED DATA'

    IF header.clean NE 0 THEN PRINTF, unit, 'CLEANED DATA vs '+STRTRIM(header.clean,2)
    IF output_info.subtract THEN PRINTF, unit, 'PEDESTAL SUBTRACTED'
    IF output_info.shift    THEN PRINTF, unit, 'PEDESTAL SHIFTED'

    PRINTF, unit, 'NLUTS      = ', STRTRIM(N_ELEMENTS(luts),1)

    IF output_info.noutputs EQ 0 THEN BEGIN

      nstart = output_info.full_detector
      nend   = output_info.full_detector

    ENDIF ELSE BEGIN

      nstart = 0
      nend   = output_info.noutputs-1
      IF output_info.full_detector EQ header.ndetectors THEN nend = nend + 1

    ENDELSE

    FOR l = 0, N_ELEMENTS(luts)-1 DO BEGIN

      PRINTF, unit, 'LUT ' + STRTRIM(luts(l),1)

      FOR k = nstart, nend DO BEGIN

        PRINTF, unit, output_info.ptitle(k)
        PRINTF, unit, 'LOWER BIN ENERGIES [keV] = ', STRTRIM(REFORM(output_chans(0,*,k,l)),1)
        PRINTF, unit, 'UPPER BIN ENERGIES [keV] = ', STRTRIM(REFORM(output_chans(1,*,k,l)),1)

      ENDFOR

    ENDFOR

    FOR k = nstart, nend DO BEGIN

      temp_data = REFORM(output_array(k,*,*), output_info.ysamples, header.npoints)

      PRINTF, unit, output_info.ptitle(k)

      ; show channels

      PRINTF, unit, header.xtitle + '    LUT     DATA_QUALITY_FLAG   /CR DATA'
      PRINTF, unit, ztitle
      FOR kk = 0, header.npoints-1 DO BEGIN

        PRINTF, unit, anytim2cal(data(kk).time), ' ', STRTRIM(FIX(data(kk).lut),1), ' ', STRTRIM(FIX(data(kk).flag),1)
        PRINTF, unit, temp_data(*,kk)

      ENDFOR

    ENDFOR

  ENDIF ELSE PRINT,'WARNING, use SPECTRAL or LINEPLOT and ENERGY vs DETECTOR format only for text output'

  ; finish off

  FREE_LUN, unit

ENDING:

  ; set up info display

  WIDGET_CONTROL, ids.infoid, SET_VALUE=STRING(10B)+header.title, /APPEND

  RETURN

IOERROR:

  WIDGET_CONTROL, ids.infoid, SET_VALUE=[STRING(10B),!ERR_STRING], /APPEND

  MESSAGE, !ERR_STRING, /CONT, /TRACEBACK

  FREE_LUN, unit

  RETURN

END

