; $Id: ies_tslider.pro,v 1.2 2004/02/26 01:53:56 friedel Exp $
; Copyright (c) 1992-1993, Research Systems, Inc.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	IES_TSLIDER
;
; PURPOSE:
;	The standard slider provided by the WIDGET_SLIDER() function is
;	integer only. This compound widget provides a time
;	slider which has CDS_INT_TIME as input.
;
; CATEGORY:
;	Compound widgets.
;
; CALLING SEQUENCE:
;	widget = IES_TSLIDER(Parent)
;
; INPUTS:
;       Parent:		The ID of the parent widget.
;
; KEYWORD PARAMETERS:
;	DRAG:		Set this keyword to zero if events should only
;			be generated when the mouse is released. If it is
;			non-zero, events will be generated continuously
;			when the slider is adjusted. Note: On slow systems,
;			/DRAG performance can be inadequate. The default
;			is DRAG=0.
;       EDIT:		Set this keyword to make the slider label be
;			editable. The default is EDIT=0.
;	FRAME:		Set this keyword to have a frame drawn around the
;			widget. The default is FRAME=0.
;	MAXIMUM:	The maximum value of the slider in CDS_INT_TIME format. 
;	MINIMUM:	The minimum value of the slider in CDS_INT_TIME format.
;	SUPPRESS_VALUE:	If true, the current slider value is not displayed.
;			The default is SUPPRESS_VALUE=0.
;	TITLE:		The title of slider. (The default is no title.)
;	UVALUE:		The user value for the widget.
;	VALUE:		The initial value of the slider in CDS_INT_TIME format.
;	XSIZE:		For horizontal sliders, sets the length.
;	YSIZE:		For vertical sliders, sets the height.
;
; OUTPUTS:
;       The ID of the created widget is returned.
;
; SIDE EFFECTS:
;	This widget generates event structures containing a field
;	named value when its selection thumb is moved. This is a
;	time value.
;
; PROCEDURE:
;	WIDGET_CONTROL, id, SET_VALUE=value can be used to change the
;		current value displayed by the widget.
;
;	WIDGET_CONTROL, id, GET_VALUE=var can be used to obtain the current
;		value displayed by the widget.
;
; MODIFICATION HISTORY:
;	April 2, 1992, SMR and AB
;		Based on the RGB code from XPALETTE.PRO, but extended to
;		support color systems other than RGB.
;	5 January 1993, Mark Rivers, Brookhaven National Labs
;		Added EDIT keyword. 
;       7 April 1993, AB, Removed state caching.
;	28 July 1993, ACY, set_value: check labelid before setting text.
;       4/3/96, MKc adapted from cw_fslider for time display.
;                  Added calls to : anytim2cal, str2utc.
;       25/9/96, MKC, modified event function S>,M>,H> LONG(val) -> LONG(val > state.bot)
;-


PRO ies_tslider_set_value, id, val

  ; Set the value of both the slider and the label
  ON_ERROR, 2						;return to caller

  stash = WIDGET_INFO(id, /CHILD)
  WIDGET_CONTROL, stash, GET_UVALUE=state, /NO_COPY

  ; allow so that max value can be set

  IF val GE state.top THEN val = state.top+1

  WIDGET_CONTROL, state.slideid, SET_VALUE = (LONG(state.top)+1) < LONG(val) > LONG(state.bot) 

  IF (state.labelid NE 0) THEN WIDGET_CONTROL, state.labelid, SET_VALUE = anytim2cal(state.top < val > state.bot,FORM=9)

  WIDGET_CONTROL, stash, SET_UVALUE=state, /NO_COPY

END



FUNCTION ies_tslider_get_value, id

  ; Return the value of the slider
  ON_ERROR, 2						;return to caller

  stash = WIDGET_INFO(id, /CHILD)
  WIDGET_CONTROL, stash, GET_UVALUE=state, /NO_COPY

  ; Get the non-adjusted value

  WIDGET_CONTROL, state.slideid, GET_VALUE = val

  val = DOUBLE(val(0))

  val = state.top < val > state.bot

  WIDGET_CONTROL, stash, SET_UVALUE=state, /NO_COPY 

  return, val

END


;-----------------------------------------------------------------------------

FUNCTION ies_tslide_event, ev

  WIDGET_CONTROL, ev.id, GET_UVALUE = uvalue

  ; Retrieve the structure from the child that contains the sub ids
  parent=ev.handler
  stash = WIDGET_INFO(parent, /CHILD)
  WIDGET_CONTROL, stash, GET_UVALUE=state, /NO_COPY

  drag = 0

  ; Get the non-adjusted value

  WIDGET_CONTROL, state.slideid, GET_VALUE = val

  val = DOUBLE(val(0))

  ; See which widget was adjusted, the slider or the label

  CASE uvalue OF

    'SLIDER' : BEGIN

                 ; slider widget

                 val = state.top < val > state.bot

                 drag = ev.drag
                  
               END

    '<S' : BEGIN

                ; left sec button 

                ; adjust to second boundaries

                val = val - 1.0 

                tmp = tai2utc ( val, /EXTERNAL )

                tmp.millisecond = 0
 
                val = utc2tai (tmp)

                val = state.top < val > state.bot

                ;Update the slider, set new value

                WIDGET_CONTROL, state.slideid, SET_VALUE = LONG(val) 

              END

    '<M' : BEGIN

                ; left min button 

                ; adjust to minute boundaries

                val = val - 60.0 

                tmp = tai2utc ( val, /EXTERNAL )

                tmp.millisecond = 0
                tmp.second      = 0
 
                val = utc2tai (tmp)

                val = state.top < val > state.bot

                ;Update the slider, set new value

                WIDGET_CONTROL, state.slideid, SET_VALUE = LONG(val) 

              END

    '<H' : BEGIN

                ; left hour button 
                ; adjust to hour boundaries

                val = val - 3600.0 

                tmp = tai2utc ( val, /EXTERNAL )

                tmp.millisecond = 0
                tmp.second      = 0
                tmp.minute      = 0
 
                val = utc2tai (tmp)

                val = state.top < val > state.bot

                ;Update the slider, set new value

                WIDGET_CONTROL, state.slideid, SET_VALUE = LONG(val) 

              END

    'S>' : BEGIN

                ; right sec button 
                ; adjust to second boundaries

                val = val + 1.0 

                tmp = tai2utc ( val, /EXTERNAL )

                tmp.millisecond = 0
 
                val = utc2tai (tmp)

                ;Update the slider, set new value

                WIDGET_CONTROL, state.slideid, SET_VALUE = LONG(val > state.bot) < (LONG(state.top)+1) 

                val = state.top < val > state.bot

             END

    'M>' : BEGIN

                ; right min button 
                ; adjust to minute boundaries

                val = val + 60.0 

                tmp = tai2utc ( val, /EXTERNAL )

                tmp.millisecond = 0
                tmp.second      = 0
 
                val = utc2tai (tmp)

                ; Update the slider, set new value

                WIDGET_CONTROL, state.slideid, SET_VALUE = LONG(val > state.bot) < (LONG(state.top)+1)

                val = state.top < val > state.bot

             END

    'H>' : BEGIN

                ; right hour button 
                ; adjust to hour boundaries

                val = val + 3600.0 

                tmp = tai2utc ( val, /EXTERNAL )

                tmp.millisecond = 0
                tmp.second      = 0
                tmp.minute      = 0
 
                val = utc2tai (tmp)

                ;Update the slider, set new value

                WIDGET_CONTROL, state.slideid, SET_VALUE = LONG(val > state.bot) < (LONG(state.top)+1)

                val = state.top < val > state.bot

             END

    'LABEL' : BEGIN  

                WIDGET_CONTROL, state.labelid, GET_VALUE = val

                val = utc2tai(str2utc(val(0)))

                ;Update the slider, set new value

                WIDGET_CONTROL, state.slideid, SET_VALUE = (LONG(state.top)+1) < LONG(val) > LONG(state.bot) 

                val = state.top < val > state.bot

              END

  ENDCASE

  ; Update label

  IF (state.labelid NE 0) THEN WIDGET_CONTROL, state.labelid, SET_VALUE=anytim2cal(val, FORM=9)

  WIDGET_CONTROL, stash, SET_UVALUE=state, /NO_COPY

  RETURN, { ID:parent, TOP:ev.top, HANDLER:0L, VALUE:val, DRAG:drag }

END

;-----------------------------------------------------------------------------

FUNCTION ies_tslider, parent, $
		DRAG = drag, $
                EDIT = edit, $
		FRAME = frame, $
		MAXIMUM = max, $
		MINIMUM = min, $
		SUPPRESS_VALUE = sup, $
		TITLE = title, $
		UVALUE = uval, $
		TOP = top,     $
		XSIZE = xsize, $
		YSIZE = ysize

  IF (N_PARAMS() EQ 0) THEN MESSAGE, 'Incorrect number of arguments', /TRACEBACK

  ON_ERROR, 2						;return to caller

  ; Defaults for keywords
  IF NOT (KEYWORD_SET(drag))  THEN drag = 0
  IF NOT (KEYWORD_SET(edit))  THEN edit = 0
  IF NOT (KEYWORD_SET(frame)) THEN frame = 0
  IF N_ELEMENTS(max) EQ 0     THEN max = 1000.0d0 
  IF N_ELEMENTS(min) EQ 0     THEN min = 0.0d0
  IF NOT (KEYWORD_SET(sup))   THEN sup = 0
  IF NOT (KEYWORD_SET(title)) THEN title = ""
  IF NOT (KEYWORD_SET(uval))  THEN uval = 0
  IF KEYWORD_SET(top) THEN val = LONG(max)+1 ELSE val = LONG(min)

  state = {slideid:0L, labelid:0L, bot:min, top:max }

  ; Motif 1.1 and newer sliders react differently to XSIZE and YSIZE
  ; keywords than Motif 1.0 or OpenLook. These defs are for horizontal sliders
  version = WIDGET_INFO(/version)
  newer_motif = (version.style eq 'Motif') and (version.release ne '1.0')

  ; The sizes of the parts depend on keywords and whether or not the
  ; float slider is vertical or horizontal
  ;these are display specific and known to be inherently evil
  sld_thk = 16
  chr_wid = 7

    ; horizontal slider

    tmp = not keyword_set(xsize)
    if (newer_motif) then begin
      if (tmp) then xsize = 0
      IF NOT (KEYWORD_SET(ysize)) THEN ysize = 0
    endif else begin
      if (tmp) then xsize = 100
      IF (TITLE NE '') THEN sld_thk = sld_thk + 21
      ysize = sld_thk		; Make the slider not waste label space
    endelse
    l_yoff = 0


  mainbase = WIDGET_BASE(parent, FRAME = frame, /ROW)

  leftbase = WIDGET_BASE(mainbase, /COLUMN)

  left = WIDGET_BUTTON ( leftbase, VALUE = "<H", UVALUE="<H" )
  left = WIDGET_BUTTON ( leftbase, VALUE = "<M", UVALUE="<M" )
  left = WIDGET_BUTTON ( leftbase, VALUE = "<S", UVALUE="<S" )

  sliderbase = WIDGET_BASE ( mainbase, /COLUMN)
  labelbase = sliderbase

  WIDGET_CONTROL, mainbase, SET_UVALUE = uval, EVENT_FUNC = 'ies_tslide_event', $
	PRO_SET_VALUE='IES_TSLIDER_SET_VALUE', $
	FUNC_GET_VALUE='IES_TSLIDER_GET_VALUE'

  IF (sup EQ 0) THEN $
    ; Only build the label if suppress_value is FALSE
    state.labelid = WIDGET_TEXT(labelbase, YOFFSET = l_yoff, $
		VALUE = anytim2cal(min<val>max,FORM=9), $
                UVALUE='LABEL', $
                edit=edit) $ 
    ELSE state.labelid = 0L

    state.slideid = WIDGET_SLIDER(sliderbase, $
		TITLE = TITLE, $
		XSIZE = xsize, $
		YSIZE = ysize, $
		/SUPPRESS_VALUE, $
		MINIMUM = LONG(min), $
		MAXIMUM = LONG(max)+1, $
		VALUE   = val, $
                UVALUE='SLIDER', $
		DRAG=drag, $
		SCROLL=1)

  rightbase = WIDGET_BASE(mainbase, /COLUMN)

  right = WIDGET_BUTTON ( rightbase, VALUE = "H>", UVALUE="H>" )
  right = WIDGET_BUTTON ( rightbase, VALUE = "M>", UVALUE="M>" )
  right = WIDGET_BUTTON ( rightbase, VALUE = "S>", UVALUE="S>" )

  WIDGET_CONTROL, WIDGET_INFO(mainbase, /CHILD), SET_UVALUE=state, /NO_COPY
  RETURN, mainbase

END
