;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_WIDGET_PLOT
;
; Purpose     :	Plot data in display widget.
;
; Explanation : Displays data in different draw widgets with scrolling windows.
;               Displays full range of data for first window in full window.
;               Scrolls each detector plot in synchronization with cursor position in
;               full window plot.
;               Allows selection of different plot types.
;
; Use         : < ies_widget_plot, /SAVE_CC >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : Displays plots in window.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	anytim2cal, ies_do_plot, ies_ncolors.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    : Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/1/96
;
; Modified    : Version 0.1, 19/9/96
;                 Added SECTOR vs DETECTOR plot option.
;                 Added units adjustment for showch.
;                 Increased space above full window for info display.
;                 Added detector sort.
;                 Changed bins to keV.
;                 Added definition of ztitle.
;                 Removed shrink button.
;                 Extended ies_draw_widget_block.
;                 Rearranged ztitle.
;                 Added ies_do_plot.
;               Version 0.2, 8/11/96
;                 Added PROFS keyword.
;               Version 0.3, 21/1/97
;                 Added AXES button.
;                 Removed PROFS keyword.
;                 Added PROFILES procedure.
;               Version 0.4, 12/5/97
;                 Changed so that uses header start time rather than first data point time.
;               Version 0.5, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Changed plotting of axes.
;               Version 0.6, 11/2/00
;                 Corrected bug where margin not set up properly if xtra plot included.
;                 Force defaults each time.
;                 Added /SET to ies_ncolors call to ensur enon-decomposed color used.
;                 Removed SAVE_CC from argument list. Always save coord conversion.
;               Version 0.7, 09/07/02
;                 Removed ncolors from argument lists.
;
; Version     :	Version 0.7, 09/07/02
;-
;**********************************************************

PRO ies_widget_plot

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  ; print some information

  WIDGET_CONTROL, ids.infoid, SET_VALUE='Plotting data, please wait', /APPEND

  WIDGET_CONTROL, ids.draw_ids(0), GET_VALUE=win

  WSET, win

  IF output_info.axes NE 0 THEN !P.REGION = [ 0.0, 0.4, 1.0, 1.0] ELSE !P.REGION = [ 0.0, 0.0, 1.0, 1.0]

  !P.CHARSIZE = 1.0

  ; adjust charsize for plots

  ies_charsize

  ; check type of plot required

  IF output_info.procname EQ 'SPECTRAL' OR output_info.procname EQ 'LINEPLOT' THEN BEGIN

    ; set procedure

    IF output_info.procname EQ 'SPECTRAL' THEN procedure = 'ies_sp_plot' ELSE $
    IF output_info.procname EQ 'LINEPLOT' THEN procedure = 'ies_sinplot'

    ; plot full display

    ies_do_plot, 0, procedure, xmargin, ymargin, /FULL, /WIDGET, SAVE_CC=dnfull, $
                 NO_XAXIS=output_info.axes EQ 2

    ; test cancel to bypass lshell plot

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id EQ 0 THEN BEGIN

      !P.REGION = [ 0.0, 0.0, 1.0, 0.4]

      ; plot other axes

      CASE output_info.axes OF
        1 : lshell_plot, header.tstart, ephem, ids.logoid
        2 : ies_draw_luts, data.time, data.lut, xmargin, [2,0], header.datatype,  $
                            TSTART=header.tstart, TEND=header.tstop
        ELSE :
      ENDCASE

    ENDIF ELSE res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT) ; swallow button release

    ; test cancel

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id NE 0 THEN GOTO, FINISH

    FOR kk = 0, output_info.noutputs-1 DO BEGIN

      ; test cancel

      res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

      IF res.id NE 0 THEN GOTO, FINISH

      ; do plot

      WIDGET_CONTROL, ids.draw_ids(kk+1), GET_VALUE=win

      WSET, win

      !P.REGION = [ 0.0, 0.0, 1.0, 1.0]
      !P.CHARSIZE = 1.0

      ; adjust charsize for plots

      ies_charsize

      ; plot display

      ies_do_plot, kk, procedure, temp_xmargin, temp_ymargin, /WIDGET, SAVE_CC=dnscroll

    ENDFOR

  ENDIF ELSE IF output_info.procname EQ 'PROFILES' OR output_info.procname EQ 'OVERLAY' THEN BEGIN

    ; plot full SPECTRAL display

    ies_do_plot, 0, 'ies_sp_plot', xmargin, ymargin, /FULL, /WIDGET, SAVE_CC=dnfull, $
                 NO_XAXIS=output_info.axes EQ 2

    ; test cancel to bypass lshell plot

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id EQ 0 THEN BEGIN

      !P.REGION = [ 0.0, 0.0, 1.0, 0.4]

      ; plot other axes

      CASE output_info.axes OF
        1 : lshell_plot, header.tstart, ephem, ids.logoid
        2 : ies_draw_luts, data.time, data.lut, xmargin, [2,0], header.datatype, $
                            TSTART=header.tstart, TEND=header.tstop
        ELSE :
      ENDCASE

    ENDIF ELSE res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT) ; swallow button release

    ; test cancel

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id NE 0 THEN GOTO, FINISH

  ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN

    ; plot full SPECTRAL display averaging over sectors

    ies_do_plot, 0, 'ies_sp_plot', xmargin, ymargin, /FULL, /WIDGET, /ANGLEANGLE, SAVE_CC=dnfull, $
                 NO_XAXIS=output_info.axes EQ 2

    ; test cancel to bypass lshell plot

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id EQ 0 THEN BEGIN

      !P.REGION = [ 0.0, 0.0, 1.0, 0.4]

      ; plot other axes

      CASE output_info.axes OF
        1 : lshell_plot, header.tstart, ephem, ids.logoid
        2 : ies_draw_luts, data.time, data.lut, xmargin, [2,0], header.datatype, $
                            TSTART=header.tstart, TEND=header.tstop
        ELSE :
      ENDCASE

    ENDIF ELSE res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT) ; swallow button release

    ; test cancel

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id NE 0 THEN GOTO, FINISH

  ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN

    ; plot full SPECTRAL display averaging over sectors

    ies_do_plot, 0, 'ies_sp_plot', xmargin, ymargin, /FULL, /WIDGET, /TIMELINE, SAVE_CC=dnfull, $
                 NO_XAXIS=output_info.axes EQ 2

    ; test cancel to bypass lshell plot

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id EQ 0 THEN BEGIN

      !P.REGION = [ 0.0, 0.0, 1.0, 0.4]

      ; plot other axes

      CASE output_info.axes OF
        1 : lshell_plot, header.tstart, ephem, ids.logoid
        2 : ies_draw_luts, data.time, data.lut, xmargin, [2,0], header.datatype, $
                            TSTART=header.tstart, TEND=header.tstop
        ELSE :
      ENDCASE

    ENDIF ELSE res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT) ; swallow button release

    ; test cancel

    res=WIDGET_EVENT(ids.logoid, BAD_ID=bad_id, /NOWAIT)

    IF res.id NE 0 THEN GOTO, FINISH

  ENDIF

  IF KEYWORD_SET(save_cc) THEN dnscroll = save_cc

  ; set !p.region and position back again

  !P.REGION   = 0
  !P.POSITION = 0

  ; set up info display

  ; get calendar day corresponding to midpoint

  midday = anytim2cal ( (header.tstart+header.tstop)/2, FORM = 9, /DATE)

  ; set up heading

  IF output_info.procname EQ 'SPECTRAL' THEN BEGIN
    heading  = ['SPECTRAL PLOT for ' + midday]
  ENDIF ELSE IF output_info.procname EQ 'LINEPLOT' THEN BEGIN
    heading  = ['LINE PLOT for ' + midday]
  ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN
    heading  = ['ANGLE/ANGLE plot for ' + midday]
  ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN
    heading  = ['TIMELINE plot for ' + midday]
  ENDIF ELSE BEGIN
    heading  = ['USER PLOT for ' + midday]
  ENDELSE

  WIDGET_CONTROL, ids.infoid, SET_VALUE=STRING(10B)+heading+' '+header.title, /APPEND

  RETURN

FINISH :

  IF KEYWORD_SET(save_cc) THEN dnscroll = save_cc

  ; set !p.region and position back again

  !P.REGION   = 0
  !P.POSITION = 0

  ; set up info display

  WIDGET_CONTROL, ids.infoid, SET_VALUE=STRING(10B)+'Plotting interrupted', /APPEND

END

