;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_WINDOW_PLOT
;
; Purpose     :	Plot data in display window.
;
; Explanation : Displays data from a number of detectors on display device.
;               Device may be PNG/GIF, PS (colour or black and white) or IDL window.
;               Leaves a space at top of each plot for the plot title.
;               The X axes are plotted below the plot region.
;
; Use         : < error = ies_window_plot (devname, file, /WINDOWS, /POSTSCRIPT, /PNG, /GIF, /BWPS, PG_POS=pg_pos, /NO_XAXIS, /NO_END) >
;
; Inputs      : devname : STRING standard device name.
;
; Opt. Inputs : None.
;
; Outputs     : error : INT 0 if ok, 1 if error
;               file  : STRING giving output file name
;               Displays plots in window.
;
; Opt. Outputs:	None.
;
; Keywords    : /WINDOWS    = plots output in x window.
;               /POSTSCRIPT = plots output as postscript file
;                               if string then assumes filename
;               /PNG        = plots output as PNG file
;                               if string then assumes filename
;               /GIF        = plots output as GIF file
;                               if string then assumes filename
;               /BWPS       = plots output as black and white postscript file
;                               if string then assumes filename
;               /PG_POS     = Controls placement of plot in display for full plot
;                             if nonzero then does not setup page header and footnote etc
;               /NO_XAXIS   = controls whether x axis displayed in full plot
;               /NO_END     = controls whether to finish page in full plot
;               /SHRINK     = Allows time axis to be adjusted.
;               /HEADING    = Bypasses default heading.
;               /FIXPANSIZ  = forces use of a fixed panel size
;               XMARGIN     = FLTARR[2] xmargin size in characters.
;               YMARGIN     = FLTARR[2] ymargin size in characters.
;               SETUP       = Flag indicating whether setup still required.
;               NPANELS     = no. of plot panels per page.
;               FOOTNOTE    = STRING footnote.
;               YTOP        = FLOAT position in page of top of panels.
;               YSIZ        = FLOAT size of panels.
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/1/96
;
; Modified    : Version 0.1, 19/9/96
;                 Added SECTOR vs DETECTOR plot option.
;                 Added units adjustment for showch.
;                 Added detector sort.
;                 Fixed so that does not interpolate angle/angle plots.
;                 Changed bins to keV.
;                 Moved definition of ztitle.
;                 Removed shrink button.
;                 Extended ies_draw_widget_block.
;                 Rearranged ztitle.
;                 Added TIMELINE plot.
;                 If ndetectors=0 plot full window.
;                 Added PG_POS, NO_END and NO_XAXIS keywords.
;                 Modified to allow ysamples to select channels displayed.
;               Version 0.2, MKC, 21/10/96
;                 Replaced charsize with !P.CHARSIZE
;                 Added ies_set_plot, ies_do_plot.
;                 Added ephemeris data.
;                 Added HEADING keyword.
;                 Changed to function to allow error return.
;               Version 0.3, MKC, 21/1/97
;                 Added AXES button.
;                 Added PROFILES procedure.
;                 Moved detector sort to ies extdata.
;                 Added DETECTOR vs SECTOR format.
;                 Added full_only tag to output_info.
;                 Returned margins from ies_do_plot.
;               Version 0.4, 15/4/97
;                 Changed so that DIV, CAL and RAT may be appliee to clean data.
;               Version 0.5, 2/5/97
;                 Changed full_only action.
;                 Changed so that uses header start time rather than first data point time.
;                 Added FULL_ONLY keyword to ies_set_plot.
;               Version 0.6, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Added ncolors to ies draw luts, ies_set_plot, ies_do_plot.
;                 Added keyword WYSIWYG.
;                 Reversed IES large pixels.
;                 Added COLTAB keyword.
;               Version 0.7, 10/11/98
;                 Added ies_ctitle.
;               Version 0.8, 12/11/98
;                 Modified button assignments.
;                 Rationalized use of noutputs.
;                 Set up band labels.
;               Version 0.9, 25/2/99
;                 Fixed bug with plotting over multiple pages.
;               Version 1.0, 25/3/99
;                 Added proper axes for pitch angle profile plot.
;           	Version 1.1, MKC, 30/8/00
;                 Fixed bug where if no. of energy bands = 1 then use of REFORM
;                 in calculation of average energy gives error.
;                 Changed so that uses output_info.minimum maximum without ABS.
;               Version 1.2, 8/9/00
;                 Added printout to ps and bwps.
;                 Previously have set things up so that last panel is an axes panel
;                 which has reduced size to accomodate the time axis.
;                 This causes a problem if last panel is not an axes panel.
;                 For compatability add a keyword to make all panels the same size
;                 including the last panel and leave room for the time axis.
;                 If fixpansiz used then no room for logo.
;               Version 1.3, 10/12/00
;                 Added PNG output
;               Version 1.4, 1/3/01
;                 Added XMARGIN keyword.
;                 Added SETUP keyword.
;                 Added NPANELS keyword.
;                 Added LOGO_POS keyword.
;               Version 1.5, 15/05/01
;                 Changed SETUP flag.
;                 Split out ies_end_plot.
;               Version 1.6, 20/07/01
;                 Added keyword FOOTNOTE.
;               Version 1.7, 04/02/02
;                 Moved reset of !p.region and !p.position to ies_end_plot.
;                 Added YTOP, YSIZ keywords.
;               Version 1.8, 26/03/02
;                 Added YMARGIN keyword.
;               Version 1.9, 08/05/02
;                 Cut out line in TIMELINE header.
;               Version 2.0, 09/07/02
;                 Removed ncolors and WYSIWYG from argument lists.
;                 Removed COLTAB keyword.
;                 Put onus on calling program to set up correct colour table.
;               Version 2.1, 08/02/03
;                 Added OVERLAY plots.
;               Version 2.2, 03/03/03
;                 Added CHARSIZE keyword.
;               Version 2.3, 28/03/03
;                 Changed definition of logo positions.
;               Version 2.4, 15/07/03
;                 Corrected bug in use of charsize.
;               Version 2.5, 30/09/03
;                 Added POLAR/RAPID switch to ies set plot.
;
; Version     :	Version 2.5, 30/09/03
;-
;**********************************************************

FUNCTION ies_window_plot, devname, file, WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps, $
                          PG_POS=pg_pos, NO_XAXIS=no_xaxis, NO_END=no_end, SHRINK=shrink, SETUP=setup, NPANELS=npanels, $
                          HEADING=heading, FIXPANSIZ=fixpansiz, XMARGIN=xmargin, YMARGIN=ymargin, $
                          LOGO_POS=logo_pos, FOOTNOTE=footnote, YTOP=ytop, YSIZ=ysiz, CHARSIZE=charsize

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                ids, dnfull, dnscroll, files, ephem, mf, at

  ON_IOERROR, IOERROR

  ; set default plot type

  IF NOT KEYWORD_SET(postscript) AND NOT KEYWORD_SET(png) AND NOT KEYWORD_SET(gif) AND NOT KEYWORD_SET(bwps) THEN windows=1

  ; set no_xaxis

  IF NOT KEYWORD_SET(no_xaxis) THEN no_xaxis=0

  ; setup npanels

  IF NOT KEYWORD_SET(npanels) THEN npanels = 6

  ; setup setup

  IF NOT KEYWORD_SET(setup) THEN setup = 0

  ; set band labels

  band_labels = ies_band_labels(header, output_info, output_chans)

  ; print some information

  IF N_ELEMENTS(ids) GT 0 THEN IF WIDGET_INFO(ids.infoid, /VALID_ID) THEN WIDGET_CONTROL, ids.infoid, SET_VALUE='Printing data, please wait'

  ; get calendar day corresponding to midpoint

  midday = anytim2cal ( (header.tstart+header.tstop)/2, FORM = 9, /DATE)

  ; check if page setup required
  ; ie pg_pos=0 or undefined and not prevented by setup flag
  ; or if demanded by setup flag

  IF (NOT KEYWORD_SET(pg_pos) AND NOT KEYWORD_SET(setup)) OR setup EQ 1 THEN BEGIN

    ; set output device

    IF NOT KEYWORD_SET(logo_pos) THEN BEGIN

      IF output_info.procname EQ 'PROFILES' THEN logo_pos = 2 ELSE logo_pos = 4

    ENDIF

    IF ies_set_plot( 0, devname, unit, file, LOGO=logo_pos, $
                     WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps, $
                     FULL_ONLY=output_info.full_only, CHARSIZE=charsize, $
                     POLAR=ies_instrument(header.datatype,/POLAR), RAPID=ies_instrument(header.datatype,/RAPID)) THEN GOTO, IOERROR

    ; assert no setup flag for external use

    setup=2

    ; get character sizes in normal coords

    x_chsize = get_chsize(/X)
    y_chsize = get_chsize(/Y)

    ; check if heading specified

    IF N_ELEMENTS(heading) EQ 0 THEN BEGIN

      ; set up heading

      IF output_info.procname EQ 'SPECTRAL' THEN BEGIN
        heading  = ['SPECTRAL PLOT for ' + midday]
      ENDIF ELSE IF output_info.procname EQ 'LINEPLOT' THEN BEGIN
        heading  = ['LINE PLOT for ' + midday]
      ENDIF ELSE IF output_info.procname EQ 'PROFILES' THEN BEGIN
        heading  = ['PROFILES for ' + midday]
      ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN
        heading  = ['ANGLE/ANGLE PLOT for ' + midday]
      ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN
        heading  = ['TIMELINE PLOT for ' + midday]
      ENDIF ELSE BEGIN
        heading  = ['USER PLOT for ' + midday]
      ENDELSE

      ; do title

      XYOUTS, 0.5 - STRLEN(heading)*x_chsize*1.5/2.0, 1-2*y_chsize, heading, CHARSIZE=1.5*!P.CHARSIZE, /NORMAL

      XYOUTS, 0.5 - STRLEN(header.title)*x_chsize*1.5/2.0, 1-4*y_chsize, header.title, $
              CHARSIZE=1.5*!P.CHARSIZE, /NORMAL

      IF N_ELEMENTS(ytop) EQ 0 THEN ytop = 1-4.5*y_chsize

    ENDIF ELSE BEGIN

      heading = heading + midday

      XYOUTS, 0.5 - STRLEN(heading)*x_chsize*1.5/2.0,  1-2*y_chsize, heading, CHARSIZE=1.5*!P.CHARSIZE, /NORMAL

      IF N_ELEMENTS(ytop) EQ 0 THEN ytop = 1-2.5*y_chsize

    ENDELSE

    ; do footnotes

    IF NOT KEYWORD_SET(footnote) THEN footnote = 'Page 1 : ' + ies_time(FORM=9)

    XYOUTS, 1.0 - STRLEN(footnote)*0.6*x_chsize, y_chsize*0.1, footnote, /NORMAL, CHARSIZE=0.6*!P.CHARSIZE

    footnote = strip_dirname(file)
    XYOUTS, 0.01, y_chsize*0.1, footnote, /NORMAL, CHARSIZE=0.6*!P.CHARSIZE

  ENDIF ELSE BEGIN

    ; need to ensure !P.CHARSIZE the same each time entered else y_chsize will change

    IF KEYWORD_SET(charsize) THEN !P.CHARSIZE = charsize ELSE !P.CHARSIZE = 1

  ENDELSE

  ; get character sizes in normal coords

  x_chsize = get_chsize(/X)
  y_chsize = get_chsize(/Y)

  ; check if ytop specified

  IF N_ELEMENTS(ytop) EQ 0 THEN ytop = 1-2.5*y_chsize

  ; check procedure for full display

  IF output_info.procname EQ 'SPECTRAL' OR output_info.procname EQ 'LINEPLOT' THEN BEGIN

    ; set procedure

    IF output_info.procname EQ 'SPECTRAL' THEN procedure = 'ies_sp_plot' ELSE $
    IF output_info.procname EQ 'LINEPLOT' THEN procedure = 'ies_sinplot'

    ; set up plot region parameters
    ; !P.REGION used for plots is 6*ysiz

    ; Previously have set things up so that last panel is an axes panel
    ; which has reduced size to accomodate the time axis.
    ; This causes a problem if last panel is not an axes panel.
    ; For compatability add a keyword to make all panels the same size
    ; including the last panel and leave room for the time axis.

    IF NOT KEYWORD_SET(fixpansiz) THEN BEGIN

      IF N_ELEMENTS(ysiz) EQ 0 THEN ysiz = (ytop-y_chsize*0.7)/npanels

    ENDIF ELSE BEGIN

      IF N_ELEMENTS(ysiz) EQ 0 THEN ysiz = (ytop-y_chsize*0.7-1.5*y_chsize)/npanels

    ENDELSE

    ; If window or png/gif plot then size is only two panels if full only
    ; if postscript then size is unaltered

    IF output_info.full_only AND (KEYWORD_SET(png) OR KEYWORD_SET(gif) OR KEYWORD_SET(windows)) THEN ysiz = 3*ysiz

    ; adjust charsize for plots

    !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]

    ies_charsize

    ; get new character sizes in normal coords

    x_chsize = get_chsize(/X)
    y_chsize = get_chsize(/Y)

    ; do appropriate plot

    IF output_info.full_only THEN BEGIN

      ; do full window only

      ; set up pg_pos

      pg_pos = 0

      !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]  - pg_pos*[ 0.0, ysiz, 0.0, ysiz]

      ; set x axis required, if luts then dont want x axis since will get time axis anyway
      ; NB can get into trouble with logical operations since numerical value of no_xaxis used in ies_do_axes.

      IF no_xaxis NE 0 THEN temp_no_xaxis = no_xaxis ELSE IF output_info.axes EQ 2 THEN temp_no_xaxis = 1 ELSE temp_no_xaxis = 0

      ; choose plot format

      ies_do_plot, pg_pos, procedure, xmargin ,ymargin, /FULL, NO_XAXIS=temp_no_xaxis, SHRINK=shrink

      ; check if other axes required

      IF no_xaxis EQ 0 THEN BEGIN

        IF output_info.axes EQ 1 THEN BEGIN

          ; define plot region for axes

          !P.REGION = !P.REGION - [0.0, ysiz, 0.0, ysiz+1.5*y_chsize]

          ; do other axes

          lshell_plot, header.tstart, ephem

        ENDIF ELSE IF output_info.axes EQ 2 THEN BEGIN

          ; define plot region for axes

          !P.REGION = !P.REGION - [0.0, ysiz-1.5*y_chsize, 0.0, ysiz]

          ; plot luts

          ies_draw_luts, data.time, data.lut, xmargin, ymargin, header.datatype, $
                         SHRINK=shrink, TSTART=header.tstart, TEND=header.tstop

        ENDIF

      ENDIF

    ENDIF ELSE IF N_ELEMENTS(pg_pos) NE 0 THEN BEGIN

      ; do page display

      !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]  - pg_pos*[ 0.0, ysiz, 0.0, ysiz]

      ; set x axis required, if luts then dont want x axis since will get time axis anyway
      ; NB can get into trouble with logical operations since numerical value of no_xaxis used in ies_do_axes.

      IF no_xaxis NE 0 THEN temp_no_xaxis = no_xaxis ELSE IF output_info.axes EQ 2 THEN temp_no_xaxis = 1 ELSE temp_no_xaxis = 0

      ; choose plot format

      ies_do_plot, pg_pos, procedure, xmargin ,ymargin, /FULL, NO_XAXIS=temp_no_xaxis, SHRINK=shrink

      ; check if other axes required

      IF no_xaxis EQ 0 THEN BEGIN

        IF output_info.axes EQ 1 THEN BEGIN

          ; define plot region for axes

          !P.REGION = !P.REGION - [0.0, ysiz, 0.0, ysiz+1.5*y_chsize]

          ; do other axes

          lshell_plot, header.tstart, ephem

        ENDIF ELSE IF output_info.axes EQ 2 THEN BEGIN

          ; define plot region for axes

          !P.REGION = !P.REGION - [0.0, ysiz-1.5*y_chsize, 0.0, ysiz]

          ; plot luts

          ies_draw_luts, data.time, data.lut, xmargin, ymargin, header.datatype, $
                         SHRINK=shrink, TSTART=header.tstart, TEND=header.tstop

        ENDIF

      ENDIF

    ENDIF ELSE BEGIN

      ; set max data panels
      ; NB need to adjust sizes to use panel 6

;      IF output_info.axes EQ 0 THEN npanels = 6 ELSE npanels = 5
      npanels = 5

      ; loop over detectors

      FOR pg_pos = 0, (output_info.noutputs < npanels)-1 DO BEGIN

        !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]  - pg_pos*[ 0.0, ysiz, 0.0, ysiz]

        ; choose plot format

        ies_do_plot, pg_pos, procedure, xmargin ,ymargin, SHRINK=shrink, $
                     NO_XAXIS=(pg_pos NE (output_info.noutputs<npanels)-1) OR (output_info.axes EQ 2)

      ENDFOR

      ; check if other axes required

      IF output_info.axes EQ 1 THEN BEGIN

        ; define plot region for axes

        !P.REGION = !P.REGION - [0.0, ysiz, 0.0, ysiz+1.5*y_chsize]

        ; do other axes

        lshell_plot, header.tstart, ephem

      ENDIF ELSE IF output_info.axes EQ 2 THEN BEGIN

        ; define plot region for axes

        !P.REGION = !P.REGION - [0.0, ysiz-1.5*y_chsize, 0.0, ysiz]

        ; plot luts

        ies_draw_luts, data.time, data.lut, xmargin, ymargin, header.datatype, $
                         SHRINK=shrink, TSTART=header.tstart, TEND=header.tstop

      ENDIF

      IF output_info.noutputs GT npanels THEN BEGIN

        ; set output device

        IF ies_set_plot( 1, devname, unit, file, WINDOWS=windows, POSTSCRIPT=postscript, $
                         PNG=png, GIF=gif, BWPS=bwps, FULL_ONLY=output_info.full_only, LOGO=logo_pos, CHARSIZE=charsize, $
                         POLAR=ies_instrument(header.datatype,/POLAR), RAPID=ies_instrument(header.datatype,/RAPID)) THEN GOTO, IOERROR

        ; second page

        ; set character sizes for titles etc

        IF KEYWORD_SET(png) OR KEYWORD_SET(gif) THEN !P.CHARSIZE = 0.75 ELSE !P.CHARSIZE = 1.0

        ; do title

        stitle = 'Continued ...'

        XYOUTS, 0.5 - STRLEN(stitle)*x_chsize*1.5/2.0, 1-2*y_chsize, stitle, CHARSIZE=1.5*!P.CHARSIZE, /NORMAL

        ; do footnote

        footnote = 'Page 2 : ' + ies_time(FORM=9)

        XYOUTS, 1.0 - STRLEN(footnote)*x_chsize*0.6, y_chsize*0.1, footnote, /NORMAL, CHARSIZE=0.6*!P.CHARSIZE

        footnote = strip_dirname(file)
        XYOUTS, 0.01, y_chsize*0.1, footnote, /NORMAL, CHARSIZE=0.6*!P.CHARSIZE

        ; adjust charsize for plots

        !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]

        ies_charsize

        ; get new character sizes in normal coords

        x_chsize = get_chsize(/X)
        y_chsize = get_chsize(/Y)

        ; loop over detectors

        FOR pg_pos = npanels, output_info.noutputs-1 DO BEGIN

          !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]  - (pg_pos-npanels)*[ 0.0, ysiz, 0.0, ysiz]

          ; choose plot format

          ies_do_plot, pg_pos, procedure, xmargin ,ymargin, SHRINK=shrink, $
                    NO_XAXIS=(pg_pos NE (output_info.noutputs-1)) OR (output_info.axes EQ 2)

        ENDFOR

        ; do other axes required

        IF output_info.axes EQ 1 THEN BEGIN

          ; define plot region for axes

          !P.REGION = !P.REGION - [0.0, ysiz, 0.0, ysiz+1.5*y_chsize]

          ; do other axes

          lshell_plot, header.tstart, ephem

        ENDIF ELSE IF output_info.axes EQ 2 THEN BEGIN

          ; define plot region for axes

          !P.REGION = !P.REGION - [0.0, ysiz-1.5*y_chsize, 0.0, ysiz]

          ; plot luts

          ies_draw_luts, data.time, data.lut, xmargin, ymargin, header.datatype, $
                         SHRINK=shrink, TSTART=header.tstart, TEND=header.tstop

        ENDIF

      ENDIF

    ENDELSE

  ENDIF ELSE IF output_info.procname EQ 'PROFILES' THEN BEGIN

    ; do profile plots

    IF NOT output_info.full_only THEN BEGIN

      ; output time

      time = 'TIME = '+anytim2cal(output_info.time)

      XYOUTS, 0.5 - STRLEN(time)*x_chsize*1.5/2.0, 1-6*y_chsize, time, CHARSIZE=1.5*!P.CHARSIZE, /NORMAL

      ytop = 1-6.3*y_chsize

      IF output_info.noutputs LE 1 THEN BEGIN
        ysiz = 0.7
        ncolumns = 1
      ENDIF ELSE IF output_info.noutputs LE 2 THEN BEGIN
        ysiz = 0.45
        ncolumns = 1
      ENDIF ELSE IF output_info.noutputs LE 3 THEN BEGIN
        ysiz = 0.3
        ncolumns = 1
      ENDIF ELSE IF output_info.noutputs LE 4 THEN BEGIN
        ysiz = 0.45
        ncolumns = 2
      ENDIF ELSE IF output_info.noutputs LE 6 THEN BEGIN
        ysiz = 0.3
        ncolumns = 2
      ENDIF ELSE IF output_info.noutputs LE 8 THEN BEGIN
        ysiz = 0.225
        ncolumns = 2
      ENDIF ELSE BEGIN
        ysiz = 0.18
        ncolumns = 2
      ENDELSE

      xsiz = 1.0/ncolumns

      ; plot profiles

      ies_plot_profiles, xsiz, ysiz, ytop, ncolumns

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'OVERLAY' THEN BEGIN

    ; do overlay profile plots

    IF NOT output_info.full_only THEN BEGIN

      ; output time

      time = 'TIME = '+anytim2cal(output_info.time)

      XYOUTS, 0.5 - STRLEN(time)*x_chsize*1.5/2.0, 1-6*y_chsize, time, CHARSIZE=1.5*!P.CHARSIZE, /NORMAL

      ytop = 1-6.3*y_chsize

      ysiz = 0.7
      ncolumns = 1

      xsiz = 1.0/ncolumns

      ; plot profiles

      ies_plot_profiles, xsiz, ysiz, ytop, ncolumns, /OVERLAY

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN

    ; do angle/angle plots

    IF NOT output_info.full_only THEN BEGIN

      ysiz = 0.2
      ytop = 1-6.3*y_chsize
      xsiz = (1-7*x_chsize)/4.0

      ies_plot_angleangle, x_chsize, y_chsize, xsiz, ysiz, ytop

      ; set plot region

      !P.REGION = [ 0.0, 0.0, 0.90, 0.1]

      ; define z title

      ztitle = ies_ctitle(header,output_info)

      ; draw color bar

      ies_do_xcolorbar, output_info.minimum, output_info.maximum, CTITLE=ztitle, LOG=output_info.log

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN

    ; do TIMELINE plots

    IF NOT output_info.full_only THEN BEGIN

      ysiz = 0.2
      ytop = 1-4*y_chsize
      xsiz = (1-7*x_chsize)/4.0

      ies_plot_timeline, x_chsize, y_chsize, xsiz, ysiz, ytop

      ; set plot region

      !P.REGION = [ 0.0, 0.0, 0.90, 0.1]

      ; define z title

      ztitle = ies_ctitle(header,output_info)

      ; draw color bar

      ies_do_xcolorbar, output_info.minimum, output_info.maximum, CTITLE=ztitle, LOG=output_info.log

    ENDIF

  ENDIF ; end timeline

  ; finish off

  IF NOT KEYWORD_SET(no_end) THEN ies_end_plot, file, devname, WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps

  ; set up info display

  IF N_ELEMENTS(ids) GT 0 THEN IF WIDGET_INFO(ids.infoid, /VALID_ID) THEN BEGIN
    IF N_ELEMENTS(heading) GT 0 THEN WIDGET_CONTROL, ids.infoid, SET_VALUE=STRING(10B)+heading, /APPEND
    WIDGET_CONTROL, ids.infoid, SET_VALUE=STRING(10B)+header.title, /APPEND
  ENDIF

  RETURN, 0

IOERROR:

  IF N_ELEMENTS(ids) GT 0 THEN IF WIDGET_INFO(ids.infoid, /VALID_ID) THEN WIDGET_CONTROL, ids.infoid, SET_VALUE=[STRING(10B),!ERR_STRING], /APPEND

  MESSAGE, !ERR_STRING, /CONT, /TRACEBACK

  SET_PLOT, devname

  RETURN, 1

END
