;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_XTICKS
;
; Purpose     :	Generates the number of ticks needed for a time axis.
;
; Explanation :	The default number of ticks for a plot axis may be too small for a large plot
;               viewed through a window. This generates as many ticks as is sensible given the
;               actual device size. This number may then be used to set the variable !X/Y/Z.TICKS
;
; Use         : < ies_xticks, time_start, time_end, xsize_in_ch, xticks, xminor, xtickname, xtickv, SHRINK=shrink >
;
; Inputs      : time_start : DOUBLE TAI start time of data.
;               time_end   : DOUBLE TAI end time of data.
;               xsize_in_ch: FLOAT size of display in characters.
;
; Opt. Inputs : None.
;
; Outputs     : xticks     : INT no. of major ticks
;               xminor     : INT no. of minor ticks
;               xtickname  : STRARR tick labels
;               xtickv     : FLTARR tick positions in data coords.
;
; Opt. Outputs: None.
;
; Keywords    : SHRINK      : INT flag indicating whether to allow x axis space around plotted data
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, MKC, 28/10/96.
;                 Added 1800. secs to possible minor tick intervals.
;                 Changed major tick interval test from 128 characters to 84 characters
;               Version 0.2, MKC, 5/8/98
;                 Changed so that month major tick marks begin on year boundary.
;                 Changed so that will give year when crossing year boundary.
;               Version 0.3, MKC, 11/6/99
;                 Changed so that long time spans handled better.
;               Version 0.4, 13/12/99
;                 Checked for Y2K compatability.
;               Version 0.5, 10/2/01
;                 Checked for zero drange.
;                 Corrected precision bug with tick_units
;               Version 0.6, 11/11/01
;                 Corrected bug for exactly 3 year interval
;
; Version     :	Version 0.6, 11/11/01
;-
;**********************************************************

PRO ies_xticks, time_start, time_end, xsize_in_ch, xticks, xminor, xtickname, xtickv, SHRINK=shrink

drange   = ABS(time_end - time_start)

; ensure drange NE 0

IF drange LT 0.1 THEN BEGIN

  drange = 0.1

  time_end = time_start+drange

ENDIF

;
; find appropriate units to have from 2 to 6 tick intervals
;
;tick_units=[10^findgen(5)*1.0d-3,10^findgen(4)*.002,10^findgen(4)*.005,$
;20.,30.,60.,60.*[2,4,5,6,10,15,20,30,60],3600.*[2.+findgen(5),8.,10.,12.],$
;86400.*[findgen(6)+1.,10.,20.,30.,60.]]

; remove 5 and 10 hour possibilities

tick_units=[10^findgen(5)*1.0d-3,10^findgen(4)*.002,10^findgen(4)*.005,$
20.,30.,60.,60.*[2,4,5,6,10,15,20,30,60],3600.*[2.,3.,4.,6.,8.,12.],$
86400.*[findgen(6)+1.,10.,20.,30.,60.]]

;
;  possible time intervals between minor ticks
;
; removed possibility of .0001,.0002,.001,.002 because can overflow LONG
; minor_units=[.0001,.0002,.001,.002,.01,.02,.1,.2,1.0,2.0,5.0,10.,20.,60.,$

minor_units=[.01,.02,.1,.2,1.0,2.0,5.0,10.,20.,60.,$
      120.,300.,600.,1800.,3600.,7200.,14400.,3600.*[6.,12.],$
      86400.*[1.,2.,5.,10.]]

;
; the units should be selected so that there are from 2 to 6 tick intervals
;
;
; is range  < 2 months?

if drange le 62*86400. then begin

   ; get units such that there are between 2 and 6 tick units per 84 characters
   ; previously was 128 characters

   nrange = drange*84.0/xsize_in_ch

   w_ok = where( (nrange/tick_units lt 7) and (nrange/tick_units ge 2))

   ; find unit with maximum no. of ticks in range

   m_ok=max(nrange/tick_units(w_ok), pos)

   i_ok=w_ok(pos)

   tick_unit=tick_units(i_ok)
;
;which minors divide evenly into major tick intervals
; this doesnt quite do - get oddities for tick_unit=0.5 for example
;   wminors=where( tick_unit mod minor_units eq 0)
; NB tick_units have precision error

  wminors = WHERE ( ABS(minor_units*ROUND(tick_unit/minor_units) - tick_unit) LT 1d-6*tick_unit )

;possible #minor intervals
;
   nminors=tick_unit/minor_units(wminors)
;
;get as close as possible to 4 sub-intervals
;
   test_minors=min(abs(nminors-4), pos)
   xminor=nminors(pos) ;number of minor tick intervals
;
; set the tick values and names
;
; get time zero to avoid problem with leap seconds
; set to 1 jan or 1 jul

    tmp       = tai2utc(time_start,/external)
    tmp.month = tmp.month < 7
    tmp.day   = 1
    tmp.hour  = 0
    tmp.minute = 0
    tmp.second = 0
    tmp.millisecond = 0

    time_zero = utc2tai ( tmp)

; looking for frac. at start
;
  new_time_start  = time_zero + LONG((time_start-time_zero)/tick_unit) * tick_unit
;
; looking for frac. at end
;
  new_time_end  = time_zero + LONG((time_end-time_zero)/tick_unit) * tick_unit

  IF new_time_end LT time_end THEN new_time_end = new_time_end + tick_unit

  xticks=ROUND((new_time_end-new_time_start)/tick_unit) ;number of tick intervals

  tic_n_range = new_time_start + FINDGEN(xticks+1)*tick_unit

endif else begin
;
; range is > 2 months
;
;  but range < 3 yrs
;
   if drange lt 1096.*86400. then begin

     ; set time start to 1st of the month

     new_time_start = tai2utc(time_start,/external)
     new_time_start.day   = 1
     new_time_start.hour  = 0
     new_time_start.minute = 0
     new_time_start.second = 0
     new_time_start.millisecond = 0

     new_time_start = utc2tai ( new_time_start)

     ; set time end to 1st of the following month

     new_time_end = tai2utc(time_end,/external)
     new_time_end.month = new_time_end.month + 1
     new_time_end.day   = 1
     new_time_end.hour  = 0
     new_time_end.minute = 0
     new_time_end.second = 0
     new_time_end.millisecond = 0

     new_time_end = utc2tai ( new_time_end)

     ; set ticks to 1st day of month in range

     month = new_time_start
     months = month

     WHILE month LT new_time_end DO BEGIN
       month = tai2utc(month,/external)
       month.month = month.month + 1
       IF month.month EQ 13 THEN BEGIN
         month.year  = month.year + 1
         month.month = 1
       ENDIF
       month = utc2tai (month)
       months = [months,month]
     ENDWHILE

     tick_unit=30.*86400.d0  ; approximately
;
;    make sure minor tick marks lie close to month boundaries

;    number of months between ticks
;    set minor tick marks close to months or years
;
     xminor = ( [1,1,1,1,1,1,2,2,2,2,2,3, $
                 3,3,3,4,4,4,4,4,4,4,4,6, $
                 6,6,6,6,6,6,6,6,6,6,6,6,6] )[N_ELEMENTS(months)-1]

;    make sure hit any year boundaries

     offset = tai2utc(new_time_start,/external)
     offset = offset.month-1
;
;    use only every dtick month/yr
;
     tic_n_range = months[WHERE( ( (INDGEN(N_ELEMENTS(months))+offset) MOD xminor) EQ 0)]

   endif else begin
;
     ; set time start to 1st of the year

     new_time_start = tai2utc(time_start,/external)
     new_time_start.month = 1
     new_time_start.day   = 1
     new_time_start.hour  = 0
     new_time_start.minute = 0
     new_time_start.second = 0
     new_time_start.millisecond = 0

     new_time_start = utc2tai ( new_time_start)

     ; set time end to 1st of the following year

     new_time_end = tai2utc(time_end,/external)
     new_time_end.year  = new_time_end.year + 1
     new_time_end.month = 1
     new_time_end.day   = 1
     new_time_end.hour  = 0
     new_time_end.minute = 0
     new_time_end.second = 0
     new_time_end.millisecond = 0

     new_time_end = utc2tai ( new_time_end)

     ; set ticks to 1st day of year in range

     year  = new_time_start
     years = year

     WHILE year LT new_time_end DO BEGIN
       year = tai2utc(year,/external)
       year.year = year.year + 1
       year = utc2tai (year)
       years = [years,year]
     ENDWHILE

     tick_unit=365.25*86400.d0  ; approximately
;
;     range is >  3 years
;
      tic_n_range=years
      xminor = ([4,4,4,4,4,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1])[N_ELEMENTS(tic_n_range)-1]

   endelse

endelse

;
; Make sure ticks are within range by both methods of time selection
;
; how many possible ticks
;

IF KEYWORD_SET(shrink) THEN BEGIN

  ; use new time interval

  num_yymm = N_ELEMENTS(tic_n_range)
  wyymm = INDGEN(num_yymm)

ENDIF ELSE BEGIN

  ; adjust no. of ticks to original interval

  wyymm=where( (tic_n_range ge time_start) and $
               (tic_n_range le time_end), num_yymm)

ENDELSE

;
;  set tick interval and number
;
xticks=n_elements(wyymm)-1
xtickv=tic_n_range(wyymm)
xtickname = STRARR(N_ELEMENTS(xtickv))

;
; set the tickname strings
;
for i=0,n_elements(xtickv)-1 do begin
   if (xtickv(i) le new_time_end) then begin
      xtickname(i)=utc2str(tai2utc(xtickv(i)))

; below gave problem with small baselength plot ie 500 secs spanning 31/12 to 1/1
;
;  fiddle  (to eliminate spurious 31-Dec entries)
;
;      if strmid(xtickname(i),5,5) eq '12-31' then begin
;         xtickv(i) = xtickv(i) + 86400.d0
;         xtickname(i)=utc2str(tai2utc(xtickv(i)))
;;      endif
;
;  even bigger fiddle on long time base plots which should never have
;  tick marks labelled 'Dec' something
;
;      if drange/86400.d0/365.25 gt 5.0 then begin
;         while (strmid(xtickname(i),5,2) eq '12') do begin
;            xtickv(i) = xtickv(i) + 86400.d0
;            xtickname(i)=utc2str(tai2utc(xtickv(i)))
;         endwhile
;      endif
   endif
endfor

;
;  determine proper label format
;
;
;SET LABEL FORMAT BASED ON SIZE OF UNITS
;
; change so that uses form mmm-yy when tick unit greater than a month

test_units=[365/2.*86400.,86400.,60.,1., .1, .01,.001]

;
;  cds format is yyyy-mm-ddThh:mm:ss.xxxZ so transform
;
mon = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
xt = xtickname
newxt = strarr(n_elements(xtickv))
for i=0,n_elements(xtickv)-1 do begin
   new = '                                     '
   m = mon(fix(strmid(xt(i),5,2))-1)
   strput,new,strmid(xt(i),8,2),0
   strput,new,'-',2
   strput,new,m,3
   strput,new,'-',6
   strput,new,strmid(xt(i),2,2),7
   strput,new,strmid(xt(i),11,12),10
   newxt(i) = new
endfor

cds_name = xtickname
xtickname = newxt

if drange/86400.d0/365.25 gt 5.0 then begin

   ; if range is greater than 5 years then just use form '1996'

   xtickname = strmid(cds_name,0,4)

endif else begin

;
; New Form dd-mmm-yy hh:mm:ss.xxx
;          0123456789012345678901
;
;          * mmm-yy
;                  * dd-mmm
;                        * hh:mm
;                              * mm:ss.x
;                                  * ss.xx
;                                    * ss.xxx
   rhs=       [8,          5,     14, 17, 19, 20, 21]
   lhs=       [3,          0,     10, 10, 13, 16, 16]

   ; get dd-mmm

   dd_mmm_yy = STRMID(xtickname,0,9)

   ; find point where the tick unit is greater than test unit
   ; i.e 0 = 6 months => mmm-yy
   ;     1 = 1 day    => dd-mmm
   ;     2 = 1 hour   => hh:mm
   ;     3 = 1 sec    => hh:mm:ss
   ;     4 = 100 msec => mm:ss.x
   ;     5 = 10 msec  => ss.xx
   ;     6 = 1 msec   => ss.xxx

   wtest=where( tick_unit ge test_units)
   lbl=[lhs(wtest(0)) ,rhs(wtest(0))]

   xtickname=strmid(xtickname,lbl(0),lbl(1)-lbl(0)+1)

   ; fix to add day to midnight boundaries when hh:mm displayed

   IF wtest(0) EQ 2 THEN BEGIN

     ; locate any 00:00 times

     list = WHERE ( xtickname EQ '00:00', count )

     IF count GT 0 THEN xtickname(list)= STRMID(dd_mmm_yy(list),0,6) + ' ' + xtickname(list)

   ENDIF

   ; fix to add year to 01-Jan form

   list = WHERE(xtickname EQ '01-Jan', count)

   IF count GT 0 THEN xtickname(list) = xtickname(list) + STRMID(dd_mmm_yy(list),6,3)

endelse

; adjust so that time_start corresponds to zero in plot

xtickv = xtickv - time_start

; get new plot range

IF KEYWORD_SET(shrink) THEN BEGIN

  time_start = xtickv(0)
  time_end   = xtickv(xticks)

ENDIF ELSE BEGIN

  time_end   = time_end - time_start
  time_start = 0.0d0

ENDELSE

END
