;+
; Project     :	POLAR - CEPPAD
;
; Name        :	eph_alt_range
;
; Purpose     :	Finds indices corresponding to a particular range around minima in alt data.
;               
; Explanation :        
;
; Use         : < markers = eph_alt_range ( ephem ) >
;
; Inputs      : ephem          : structure giving ephemeris details.
;
; Opt. Inputs : None.
;
; Outputs     : markers        : DBLARR(2,n) giving time bands around the one or two alt data minima in seconds
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	utc2tai, anytim2utc.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None..
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/5/97
;
; Modified    :	Version 0.1, MKC, 4/11/98
;                 Updated to use new ephemeris structure.
;
; Version     :	Version 0.1, 4/11/98
;-
;**********************************************************

FUNCTION eph_alt_range, ephem

  ; get altitude in earth radii

  alt = ephem.r/6378.

  ; get time in hours

  sut = (ephem.time - utc2tai(anytim2utc(anytim2cal(ephem[0].time,/DATE))))/3600.

  ; get no of points
  
  npoints = N_ELEMENTS(alt)

  ; find minima in alt data
  ; in a 24 hour period there will always be at least on true maximum and one true minimum
  ; there may also be an additional maximum or minimum (but not both)

  min_alt0 = MIN(alt, min_pos0)

  ; find time of minimum in hours

  times = sut(min_pos0)

  ; find possible other minimum 

  IF min_pos0 LT npoints/2 THEN BEGIN

    min_alt1 = MIN(alt(min_pos0+npoints/4:*), min_pos1)

    min_pos1 = min_pos1 + min_pos0 + npoints/4

    IF min_alt1 LT 1.01*min_alt0 THEN times = [times,sut(min_pos1)]

  ENDIF ELSE BEGIN

    min_alt1 = MIN(alt(0:min_pos0-npoints/4), min_pos1)

    IF min_alt1 LT 1.01*min_alt0 THEN times = [sut(min_pos1), times]

  ENDELSE

  ; fix time period around alt minima and orbit period (hours)

  tminus  = 4.0
  tplus   = 4.0

  ; get period around minima

  times = TRANSPOSE([[times-tminus], [times+tplus]])

  RETURN, utc2tai(anytim2utc(anytim2cal(ephem[0].time,/DATE))) + times*3600.0d0

END


