;+
; Project     :	POLAR - CEPPAD
;
; Name        :	eph_lsh_range
;
; Purpose     :	Finds interval from minimum L shell to supplied L shell on north pole side.
;
; Explanation : L shell varies too fast over south pole. returns interval from minimum L shell
;               to supplied L shell on north pole side.
;
; Use         : < list = eph_lsh_range ( lshells, lthreshold_north) >
;
; Inputs      : lshells            : lshell values.
;               lthreshold_north   : integer L shell threshold defining bands on north side
;
; Opt. Inputs : None.
;
; Outputs     : list         : DBLARR(3,n) giving indices around L shell ranges and orbit type
;                              0 = lhs
;                              1 = rhs
;                              2 = type
;                              If invalid L shell values then returns -1.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None..
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/8/97
;
; Modified    :	Version 0.1, 10/9/98, MKC
;                 Used fixed south side threshold.
;               Version 0.2, 20/07/01
;                 Used ies_runs.
;
; Version     :	Version 0.2, 20/07/01
;-
;**********************************************************

FUNCTION eph_lsh_range, lshells, lthreshold_north

  ; fix south pole threshold

  lthreshold_south = lthreshold_north < 20

  ; get no of points

  npoints = N_ELEMENTS(lshells)

  ; get list of threshold crossings

  right_list = ies_runs(lshells LT lthreshold_south)

  count = N_ELEMENTS(right_list)

  ; get only runs below threshold

  IF lshells(0) LT lthreshold_south THEN BEGIN

    ; use even runs

    left_list = [0]

    IF count GT 2 THEN left_list = [left_list, right_list(1+2*INDGEN((count-1)/2)) + 1]

    right_list = right_list(2*INDGEN((count+1)/2))

  ENDIF ELSE BEGIN

    ; use odd runs

    ; return -1 if no runs below L threshold

    IF count EQ 1 THEN BEGIN
      PRINT, 'No runs below L threshold'
      RETURN, -1
    ENDIF

    left_list = right_list(2*INDGEN(count/2)) + 1

    right_list = right_list(1+2*INDGEN(count/2))

  ENDELSE

  ; loop through intervals finding minimum l shell and adjusting south side

  ; determine where we are ignoring truncated orbits

  first = 0
  last = N_ELEMENTS(right_list)-1
  IF left_list(0)  EQ 0 THEN first = 1
  IF right_list(last) EQ npoints-1 THEN last = last-1
  orbits = INTARR(N_ELEMENTS(right_list))

  ; check at least one good orbit

  IF last LT first THEN BEGIN
    MESSAGE, 'Invalid L shell data', /CONT
    RETURN, -1
  ENDIF

  ; loop through good orbits

  last_orbit = -1

  FOR k = first, last DO BEGIN

    dummy = MIN(lshells(left_list(k):right_list(k)), minpos)

    minpos = minpos + left_list(k)

    IF right_list(k) - minpos GT minpos - left_list(k) THEN BEGIN
      orbits(k) = 1
      left_list(k) = minpos
    ENDIF ELSE IF minpos - left_list(k) GT right_list(k) - minpos THEN BEGIN
      orbits(k) = 0
      right_list(k) = minpos
    ENDIF ELSE BEGIN
      MESSAGE, 'Anomalous L shell data', /CONT
      orbits(k) = 1 - last_orbit
      IF last_orbit EQ 0 THEN left_list(k) = minpos ELSE $
        IF last_orbit EQ 1 THEN right_list(k) = minpos
    ENDELSE

    IF orbits(k) EQ last_orbit THEN MESSAGE, 'Invalid orbit found', /CONT

    last_orbit = orbits(k)

  ENDFOR

  ; deal with rhs truncated orbit

  k = N_ELEMENTS(right_list)-1
  IF right_list(k) EQ npoints-1 THEN BEGIN
    orbits(k) = 1 - last_orbit
    dummy = MIN(lshells(left_list(k):right_list(k)), minpos)
    minpos = minpos + left_list(k)
    IF orbits(k) EQ 1 THEN BEGIN
      IF minpos EQ npoints-1 THEN BEGIN
        right_list = right_list(0:last)
        left_list  = left_list(0:last)
        orbits     = orbits(0:last)
      ENDIF ELSE left_list(k) = minpos
    ENDIF ELSE IF minpos NE npoints-1 THEN right_list(k) = minpos
  ENDIF

  ; deal with lhs truncated orbit

  k = 0
  IF left_list(0) EQ 0 THEN BEGIN
    orbits(0) = 1 - orbits(1)
    dummy = MIN(lshells(left_list(k):right_list(k)), minpos)
    minpos = minpos + left_list(k)
    IF orbits(k) EQ 0 THEN BEGIN
      IF minpos EQ 0 THEN BEGIN
        right_list = right_list(1:*)
        left_list  = left_list(1:*)
        orbits     = orbits(1:*)
      ENDIF ELSE right_list(k) = minpos
    ENDIF ELSE IF minpos GT 0 THEN left_list(k) = minpos
  ENDIF

  ; extend search through lshells

  IF lthreshold_north GT lthreshold_south THEN BEGIN

    ; loop through lists

    FOR k = 0, N_ELEMENTS(right_list)-1 DO BEGIN

      IF orbits[k] EQ 0 THEN BEGIN

        WHILE ( left_list[k] GT 0 AND lshells[left_list[k]] LT lthreshold_north ) DO $
          left_list[k] = left_list[k] - 1

        IF lshells[left_list[k]] GT lthreshold_north THEN left_list[k] = left_list[k] + 1

      ENDIF ELSE BEGIN

        WHILE ( right_list[k] LT npoints-2 AND lshells[right_list[k]] LT lthreshold_north ) DO $
          right_list[k] = right_list[k] + 1

        IF lshells[right_list[k]] GT lthreshold_north THEN right_list[k] = right_list[k] - 1

      ENDELSE

    ENDFOR

  ENDIF

  ; return list of left and right points

  RETURN, TRANSPOSE([[left_list],[right_list],[orbits]])

END


