;+
; Project     :	POLAR - CEPPAD
;
; Name        :	get_at
;
; Purpose     :	Gets attitude data.
;
; Explanation : Attitude data is available in daily files. This program gets the structure returned by
;               read_cdf_at.
;               Expects data to be in one of files :
;                       'PO_CEP_OR_DIR'/yyyy/po_at_def_yyyymmdd_vnn.cdf
;                       'PO_CEP_OR_DIR'/yyyy/po_at_pre_yyyymmdd_vnn.cdf
;                       'PO_CEP_AT_DIR'/yyyy/yymmddnn.cdf
;                       'PO_CEP_AT_DIR'/istp/po_at_def_yyyymmdd_vnn.cdf
;                       'PO_CEP_AT_DIR'/istp/po_at_pre_yyyymmdd_vnn.cdf
;                       'PO_CEP_AT_DIR'/def/yymmddnn.cdf
;                       'PO_CEP_AT_DIR'/pre/yymmddnn.cdf
;
; Use         : < at = get_at ( time, /QUIET ) >
;
; Inputs      : time       : time in any CDS format.
;
; Opt. Inputs : None.
;
; Outputs     : at          : structure giving attitude details.
;                                If none found then 0 returned.
;                                 .sut : time in decimal hours of day
;
; Opt. Outputs:	None.
;
; Keywords    : QUIET : If set suppresses printout.
;
; Calls       :	utc2tai, anytim2utc, read_cdf_at.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	Adapted from get_ephem.
;
; Written     :	Version 0.0, Martin Carter, RAL, 6/10/98
;
; Modified    :	Version 0.1, MKC, 11/5/99.
;                 Corrected pre and def file names -- was only searching istp correctly.
;         	    Version 0.2, 09/01/03.
;                 Added sub-directory ../or/2002/.. to search.
;                 Removed check for pre or def in file name.
;         	    Version 0.3, 03/02/03.
;                 Corrected bug : "Variable undefined : LIST".
;               Version 0.4, 09/10/03
;                 Used ies get dir.
;                 Changed printout.
;               Version 0.5, 19/01/04
;                 Added yyyy/po_at_def_yyyymmdd_vnn.cdf and yyyy/po_at_pre_yyyymmdd_vnn.cdf to search.
;                 Search for *.cdf rather than *.cdf*.
;
; Version     :	Version 0.5, 19/01/04
;-
;**********************************************************

FUNCTION get_at, time, QUIET=quiet

  ; convert time to external time

  ext_time = anytim2utc ( time, /EXTERNAL )

  ; check if attitude file present

  efile = STRING( ext_time.year, ext_time.month, ext_time.day, '*.cdf', FORMAT='(I4.4,I2.2,I2.2,A)' )

  count = 0

  ; look in ..AT/yyyy/po_at_*yyyymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( STRMID(efile,0,4), 'po_at_*'+efile)
    efiles = concat_dir ( ies_get_dir('AT',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..AT/yyyy/yymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( STRMID(efile,0,4), STRMID(efile,2,11))
    efiles = concat_dir ( ies_get_dir('AT',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..AT/def/yymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( 'def', STRMID(efile,2,11))
    efiles = concat_dir ( ies_get_dir('AT',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..AT/pre/yymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( 'pre', STRMID(efile,2,11))
    efiles = concat_dir ( ies_get_dir('AT',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..AT/istp/po_at_*yyyymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( 'istp', 'po_at_*'+efile)
    efiles = concat_dir ( ies_get_dir('AT',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; get attitude data structure for relevant day

  struct = 0

  IF count GT 0 THEN BEGIN

    ; read highest version number (ie last file)
    ; NB will select pre rather than def

    IF NOT KEYWORD_SET(quiet) THEN PRINT, 'Reading attitude file :'+efiles[count-1]

    struct = read_cdf_at ( efiles[count-1], TYPE=type )

    IF type EQ 'PRE' THEN PRINT, 'NOTE : Using preliminary attitude data'

    ; check attitude data structure

    IF N_TAGS(struct) EQ 0 THEN PRINT, 'WARNING, ATTITUDE DATA STRUCTURE INVALID'

  ENDIF ELSE struct=0

  ; test if structure OK

  s = size ( struct )

  IF ( N_ELEMENTS ( s ) LT 4  ) THEN BEGIN

     print, '      *** Sorry Attitude data not available!! ***'

     RETURN, 0

 endif

 IF ( N_ELEMENTS(struct) LT 2 ) THEN BEGIN

     print, '      *** Insufficient Attitude data available!! ***'

     RETURN, 0

 ENDIF

 RETURN, struct

END


