;+
; Project     :	POLAR - CEPPAD
;
; Name        :	get_ephem
;
; Purpose     :	Gets ephemeris data and performs some manipulations of L shell values.
;
; Explanation : Ephemeris data is available in daily files. This program gets the structure returned by read_cdf_ephem
;               and performs some manipulations.
;               Expects data to be in one of files :
;                       'PO_CEP_OR_DIR'/yyyy/po_or_def_yyyymmdd_vnn.cdf
;                       'PO_CEP_OR_DIR'/yyyy/po_or_pre_yyyymmdd_vnn.cdf
;                       'PO_CEP_OR_DIR'/yyyy/yymmddnn.cdf
;                       'PO_CEP_OR_DIR'/istp/po_or_def_yyyymmdd_vnn.cdf
;                       'PO_CEP_OR_DIR'/istp/po_or_pre_yyyymmdd_vnn.cdf
;                       'PO_CEP_OR_DIR'/def/yymmddnn.cdf
;                       'PO_CEP_OR_DIR'/pre/yymmddnn.cdf
;                 -- Changes MLT so that has no discontinuities at day boundaries.
;
; Use         : < ephem = get_ephem ( time, /QUIET ) >
;
; Inputs      : time       : time in CDS format.
;
; Opt. Inputs : None.
;
; Outputs     : ephem          : structure giving ephemeris details.
;                                If none found then 0 returned.
;
; Opt. Outputs:	None.
;
; Keywords    : QUIET : If set suppresses printout.
;
; Calls       :	utc2tai, anytim2utc.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	Adapted from cammice_survey.pro.
;
; Written     :	Version 0.0, Martin Carter, RAL, 22/10/96
;
; Modified    :	Version 0.1, MKC, 19/2/97.
;                 Allowed .dat, .DEF, .PRE ephemeris file types.
;         	Version 0.2, MKC, 21/8/97.
;                 Added version numbers to ephemeris file names.
;                 Added QUIET keyword.
;         	Version 0.3, MKC, 5/10/98.
;                 Return ephemeris cdf structure.
;                 Runs direct from CDF file now.
;                 Changed name of environment variable pointing to data.
;                 Returns 0 if failed.
;                 Found problem with mlt fix didnt work for days around 1 Aug 98 - 4 Aug 98
;         	Version 0.4, MKC, 11/5/99.
;                 Corrected pre and def file names -- was only searching istp correctly.
;         	Version 0.5, 09/01/03.
;                 Added sub-directory ../or/2002/.. to search.
;                 Removed check for pre or def in file name.
;         	Version 0.6, 03/02/03.
;                 Corrected bug : "Variable undefined : LIST".
;           Version 0.7, 09/10/03
;                 Used ies get dir.
;                 Changed printout.
;           Version 0.8, 19/01/04
;                 Added yyyy/po_or_def_yyyymmdd_vnn.cdf and yyyy/po_or_pre_yyyymmdd_vnn.cdf to search.
;                 Search for *.cdf rather than *.cdf*.
;
; Version     :	Version 0.8, 19/01/04
;-
;**********************************************************

FUNCTION get_ephem, time, QUIET=quiet, NFP=nfp

  ; convert time to external time

  ext_time = anytim2utc ( time, /EXTERNAL )

  ; find ephemeris cdf file present

  efile = STRING( ext_time.year, ext_time.month, ext_time.day, '*.cdf', FORMAT='(I4.4,I2.2,I2.2,A)' )

  count = 0

  ; look in ..OR/yyyy/po_or_*yyyymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( STRMID(efile,0,4), 'po_or_*'+efile)
    efiles = concat_dir ( ies_get_dir('OR',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..OR/yyyy/yymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( STRMID(efile,0,4), STRMID(efile,2,11))
    efiles = concat_dir ( ies_get_dir('OR',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..OR/def/yymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( 'def', STRMID(efile,2,11))
    efiles = concat_dir ( ies_get_dir('OR',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..OR/pre/yymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( 'pre', STRMID(efile,2,11))
    efiles = concat_dir ( ies_get_dir('OR',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; look in ..OR/istp/po_or_*yyyymmdd*.cdf

  IF count EQ 0 THEN BEGIN

    efiles = concat_dir ( 'istp', 'po_or_*'+efile)
    efiles = concat_dir ( ies_get_dir('OR',/POLAR), efiles)

    efiles = FINDFILE ( efiles, COUNT=count )

  ENDIF

  ; restore ephemeris data structure for relevant day

  ephem = 0

  IF count GT 0 THEN BEGIN

    ; read highest version number (ie last file)
    ; NB will select pre rather than def

    IF NOT KEYWORD_SET(quiet) THEN PRINT, 'Reading ephemeris file :'+efiles[count-1]

    ephem = read_cdf_ephem ( ext_time.year, ext_time.month, ext_time.day, $
                             FILE = efiles[count-1], TYPE=type )

    IF type EQ 'PRE' THEN PRINT, 'NOTE : Using preliminary ephemeris data'

    ; check ephemeris data structure

    IF N_TAGS(ephem) EQ 0 THEN PRINT, 'WARNING, EPHEMERIS DATA STRUCTURE INVALID'

  ENDIF ELSE ephem=0

  ; test if structure OK

  s_eph = size ( ephem )

  IF ( N_ELEMENTS ( s_eph ) LT 4  ) THEN BEGIN

     print, '      *** Sorry Ephemeris data not available!! ***'

     RETURN, 0

 endif
;
 IF ( N_ELEMENTS(ephem) LT 2 ) THEN BEGIN

     print, '      *** Insufficient Ephemeris data available!! ***'

     RETURN, 0

 ENDIF

 ; change mlt so that has no discontinuities at day boundaries

 mlt = ephem.edmlt
 neph = n_elements ( mlt )
 dmlt = mlt ( 1:neph-1 ) - mlt ( 0:neph-2 )
 dd = max ( abs ( dmlt ) )
 if ( dmlt ( !c ) lt 0.0 ) then mlt ( !c+1:neph-1 ) = mlt ( !c+1:neph-1 ) + 24.0
 if ( dmlt ( !c ) gt 0.0 ) then mlt ( 0:!c ) = mlt ( 0:!c ) + 24.0
 dmlt = mlt ( 1:neph-1 ) - mlt ( 0:neph-2 )
 dd = max ( abs ( dmlt ) )
 if ( dd lt 20.0 ) then goto, cont
 if ( dmlt ( !c ) lt 0.0 ) then mlt ( !c+1:neph-1 ) = mlt ( !c+1:neph-1 ) + 24.0
 if ( dmlt ( !c ) gt 0.0 ) then mlt ( 0:!c ) = mlt ( 0:!c ) + 24.0
 dmlt = mlt ( 1:neph-1 ) - mlt ( 0:neph-2 )
 dd = max ( abs ( dmlt ) )
 if ( dd lt 20.0 ) then goto, dcont
 if ( dmlt ( !c ) lt 0.0 ) then mlt ( !c+1:neph-1 ) = mlt ( !c+1:neph-1 ) + 24.0
 if ( dmlt ( !c ) gt 0.0 ) then mlt ( 0:!c ) = mlt ( 0:!c ) + 24.0
 dcont:
 if ( min ( mlt ) gt 24.0 ) then mlt = mlt - 24.0
 cont:
 ephem.edmlt = mlt

 ; check if want MFE L shell

 IF KEYWORD_SET(nfp) THEN BEGIN

   struct = get_mf(time,nfp=nfp)

   lat = 90.0
;   lat = 82.4
;   lon = -137.3
   lon = 0
   sinl = COS((struct.nfp[1,*]-lon)/180.0*!PI)*COS(struct.nfp[0,*]/180.*!PI)*COS(lat/180.*!PI) + $
            SIN(struct.nfp[0,*]/180.*!PI)*SIN(lat/180.*!PI)

   lsh = 1/(1-sinl^2)

   ephem.l = INTERPOL(lsh, struct.time,ephem.time)

 ENDIF

 RETURN, ephem

END


