;+
; Project     :	POLAR - CEPPAD
;
; Name        :	get_mf
;
; Purpose     :	Gets magnetic field data.
;
; Explanation : Magnetic field data is available in daily files. This program gets the structure returned by
;               read_cdf_mfe or from the CEPPAD HK MF files.
;               If MFE data used then the magnetic field strength is returned in nT in SC coords.
;               If HK MF data used then the magnetic field direction is returned in SC coords and
;               the field strength is normalized to 1 nT.
;
; Use         : < mf = get_mf ( time, /QUIET ) >
;
; Inputs      : time       : time in any CDS format.
;
; Opt. Inputs : None.
;
; Outputs     : mf          : structure giving magnetic field details.
;                                If none found then 0 returned.
;                                 .time : time in TAI
;
; Opt. Outputs:	None.
;
; Keywords    : QUIET  : If set suppresses printout.
;               MF     : If set does not look for MFE data.
;               FIELD  : If set does not look for MF data.
;
; Calls       :	utc2tai, anytim2utc, read_cdf_mfe.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	Adapted from get_ephem.
;
; Written     :	Version 0.0, Martin Carter, RAL, 6/10/98
;
; Modified    :	0.1 19/5/00 MKC
;                 Added KEYWORD FIELD so that only returns MFE data not MF data
;            	0.2, 15/01/02
;                 Added /CHECK keyword to restore of MF data.
;            	0.3, 06/10/03
;                 Changed HK MF structure name from CMF_STRUCT_BSC_V0 to CMF_STRUCT_BSC_XYZ_V1
;               0.4, 09/10/03
;                 Use ies get dir.
;                 Changed printout,
;               0.5, 22/10/03
;                 Added subdirectory yyyy for MFE in line with get ephem and get at.
;               0.6, 19/01/04
;                 Checked for *.cdf rather than *.cdf*
;
; Version     :	Version 0.6, 19/01/04
;-
;**********************************************************

FUNCTION get_mf, time, QUIET=quiet, MF=mf, TSY=tsy, NFP=nfp, FIELD=field

  ; convert tai time to external time

  ext_time = anytim2utc ( time, /EXTERNAL )

  struct = 0

  ; check if can use MFE data

  IF KEYWORD_SET(mf) THEN BEGIN

    count = 0

  ENDIF ELSE BEGIN

    ; check if MFE file present
    ; looking for ..MFE/yyyy/po_k0_mfe_yyyymmdd*.cdf

    efile = STRING( 'po_k0_mfe_', ext_time.year, ext_time.month, ext_time.day, '*.cdf', FORMAT='(A,I4.4,I2.2,I2.2,A)' )

    efile = concat_dir ( STRING(ext_time.year, FORMAT='(I4.4)'), efile)

    efile = concat_dir ( ies_get_dir('MFE',/POLAR), efile)

    efiles = FINDFILE ( efile, COUNT=count )

  ENDELSE

  ; get data structure for relevant day
  ; if MFE data present then use this (unless explicily forced not to)

  IF count GT 0 THEN BEGIN

    ; use MFE data
    ; read highest version number (ie last file)

    IF NOT KEYWORD_SET(quiet) THEN BEGIN
      PRINT, 'Reading magnetic field file :'+efiles[count-1]
    ENDIF

    struct = read_cdf_mfe ( efiles[count-1], TSY=tsy, NFP=nfp )

    ; check data structure

    IF N_TAGS(struct) EQ 0 THEN BEGIN
      PRINT, 'WARNING, MFE DATA STRUCTURE INVALID'
    ENDIF

  ENDIF ELSE IF (NOT KEYWORD_SET(field)) AND (NOT KEYWORD_SET(tsy)) AND (NOT KEYWORD_SET(nfp))THEN BEGIN

    ; no MFE data for use
    ; check if forced not to use

    IF NOT KEYWORD_SET(mf) THEN PRINT, '      *** Sorry MFE data not available!! ***'

    ; try to use MF data

    ; check if MF file present

    efile = STRING( 'PO_CEP_RAL_MF_', ext_time.year, ext_time.month, ext_time.day, '.DAT*', FORMAT='(A,I4.4,I2.2,I2.2,A)' )

    fln=efile
    efile = concat_dir ( ies_get_dir('MF',/POLAR), efile)

    efiles = FINDFILE ( efile, COUNT=count )

    if count eq 0 then begin ;try add year dir
        mf_dir = ies_get_dir('MF',/POLAR) + $
                     string(ext_time.year,format="(i4.4)")+'/'
        efile = concat_dir( mf_dir,fln)
        efiles = FINDFILE ( efile, COUNT=count )
        stop
    endif

    IF count EQ 1 THEN BEGIN

      IF NOT KEYWORD_SET(quiet) THEN PRINT, 'Reading magnetic field file :'+efiles[0]

      ; restore magnetic field data structure for relevant day

      input_header = 0
      input_data   = 0

      restore_file = efiles[0]

      ; check if compressed file

      IF ies_test_gzip(restore_file) THEN ies_gunzip, restore_file, /CHECK

      RESTORE, restore_file

      ; check more than one point

      IF input_header.npoints GT 1 THEN BEGIN

        ; set up structure
        ; not sure when sampled assume at midpoints else have trouble with interpolating
        ; across 360 degs.

        struct = REPLICATE({CMF_STRUCT_BSC_XYZ_V1, TIME:0.d0, BT: 0.0, MF:[0.,0.,0.]},input_header.npoints-1)

        struct.time = input_data[0:input_header.npoints-2] .time

        ; NB treat magel as polar angle

        az = REFORM(input_data[0:input_header.npoints-2].data[0,0,0]/180.*!PI)
        po = REFORM(input_data[0:input_header.npoints-2].data[0,1,0]/180.*!PI)

        struct.mf[0] = SIN(po)*COS(az)
        struct.mf[1] = SIN(po)*SIN(az)
        struct.mf[2] = COS(po)
        struct.bt = 1.

      ENDIF

    ENDIF ELSE BEGIN

      PRINT, '      *** Sorry CEPPAD MF data not available!! ***'

    ENDELSE

  ENDIF ELSE BEGIN

    ; forced to use MFE data but not available

    PRINT, '      *** Sorry MFE data not available!! ***'

  ENDELSE

  RETURN, struct

END


