;+
; Project     : CLUSTER - RAPID
;
; Name        : get_rapid_mf
;
; Purpose     : Gets FGM magnetic field data.
;
; Explanation : Magnetic field data is available in daily CDF files.
;               The magnetic field strength is returned in nT in GSE coords or SC coords.
;
; Use         : < mf = get_rapid_mf ( sc, time, /QUIET ) >
;
; Inputs      : sc  : STRING character giving sc
;               time       : time in CDS format.
;
; Opt. Inputs : None.
;
; Outputs     : mf          : structure giving magnetic field.
;                             If none found then 0 returned.
;
; Opt. Outputs:  None.
;
; Keywords    : QUIET : If set suppresses printout.
;               GSE   : FLAG indicating return MF in GSE coords
;               AT    : STRUCTURE attitude data.
;               DATA  : STRUCTURE ARRAY containing data with mvals.
;
; Written     : Version 0.0, Martin Carter, RAL, 09/05/02
;
; Modified    : Version 0.1, 30/01/03
;                 Changed directory structure to ../FGM/Cn where n is SC
;                 Removed sun sector offset adjustment.
;                 Changed structure name.
;                 Corrected bug : time not set up time in structure.
;               Version 0.2, 27/02/03
;                 Added AT keyword.
;               Version 0.3, 04/10/03
;                 Added DATA keyword.
;                 Changed structure name from FGM_SC_STRUCT_SCn_V0 to FGM_BSC_XYZ_STRUCT_SCn_V1
;               Version 0.4, 09/10/03
;                 Used ies get dir.
;
; Version     :	Version 0.4, 09/10/03
;-
;**********************************************************

FUNCTION get_rapid_mf, sc, time, AT=at, QUIET=quiet, GSE=gse, DATA=data

  ; convert time to external time

  ext_time = anytim2utc ( time, /EXTERNAL )

  ; find ephemeris cdf file name : C1_PP_FGM_yyyymmdd_Vnn.cdf

  files = STRING( 'C'+sc+'_PP_FGM_',ext_time.year, ext_time.month, ext_time.day, '_V*.cdf', FORMAT='(A,I4.4,I2.2,I2.2,A)' )

  ; look in

  files = concat_dir ( 'C'+sc, files)

  files = concat_dir ( ies_get_dir('FGM_CDF',/RAPID), files)

  files = FINDFILE ( files, COUNT=count )

  ; if no files then try mvals

  IF count EQ 0 THEN GOTO, mvals

  ; if more than one file then select last version

  IF count GT 1 THEN BEGIN

    list = SORT(files)

    files = files[list[count-1]]

  ENDIF

  ; read FGM file : fgm={time:DOUBLE,b_gse:FLTARR(3)}

  fgm = read_cdf_fgm(sc, FILES=files[0] )

  ; if no fgm then try mvals

  IF NOT KEYWORD_SET(fgm) THEN GOTO, mvals

  ; check if want GSE or SC coords

  IF KEYWORD_SET(gse) THEN BEGIN

    ; return magnetic field in GSE coords structure

    RETURN, fgm

  ENDIF ELSE BEGIN

    ; get attitude data

    IF NOT KEYWORD_SET(at) THEN at = get_rapid_aux(sc, time, QUIET=quiet, /AT)

  ; if no at then try mvals

    IF NOT KEYWORD_SET(at) THEN GOTO, mvals

    ; set up magnetic field in SC coords structure

    structure_name = 'FGM_BSC_XYZ_STRUCT_SC'+sc+'_V1'

    IF NOT EXECUTE ( 'dummy_structure={' + structure_name + ',' + $
            'time:0.0d0,'+ $
            "mf:FLTARR(3)}" ) THEN RETURN, 0

    mf = REPLICATE(dummy_structure,N_ELEMENTS(fgm))

    ; convert MF in GSE coords to SC XYZ coords

    mf.mf = ies_gse_to_sc ( at.gse_ra, at.gse_dec, at.time, fgm.mf, fgm.time, 0.0, /XYZ)

    mf.time = fgm.time

  ENDELSE

  RETURN, mf

; try using mvals to get mf

mvals :

  IF NOT KEYWORD_SET(data) THEN RETURN, 0

  mf = rcs_mag_bsc(data)

  ; issue warning

  IF KEYWORD_SET(mf) THEN MESSAGE, 'WARNING, USING IEL DATA FOR MAGNETIC FIELD DIRECTION', /INFORMATIONAL

  RETURN, mf

END


