;+
; Project     :	POLAR - CEPPAD
;
; Name        :	get_sc_mf
;
; Purpose     :	Gets magnetic field data for POLAR or CLUSTER sc in SC coords and nT.
;
; Explanation : Gets magnetic field data for day defined by midpoint of time range specified
;               by header.
;               If input already contains .mf tag then uses this.
;
; Use         : < mf = get_sc_mf ( header, data, /QUIET ) >
;
; Inputs      : header : STRUCTURE containing IF header info.
;               data   : STRUCTURE ARRAY containing IF data.
;
; Opt. Inputs : None.
;
; Outputs     : mf          : structure giving magnetic field details.
;                                If none found then 0 returned.
;                                 .time : time in TAI
;
; Opt. Outputs:	None.
;
; Keywords    : QUIET  : If set suppresses printout.
;               MF     : If set does not look for MFE data.
;               FIELD  : If set does not look for MF data.
;               AT     : STRUCTURE attitude data.
;               FORCE  : FLAG indicating should force magnetic field data to be re-read
;
; Modified    :	Version 0.1, 29/07/02
;                 Corrected bug -- tried to use structure name to differentiate
;                 between CEPPAD and RAPID but POLAR IF files do not have POLAR
;                 in structure name.
;               Version 0.2, 27/02/03
;                 Added AT keyword.
;               Version 0.3, 04/10/03
;                 Added data argument.
;               Version 0.4, 24/10/03
;                 Used .mf tag if present.
;               Version 0.5, 31/10/03
;                 Added FORCE keyword.
;
; Version     : Version 0.4, 24/10/03
;-
;**********************************************************

FUNCTION get_sc_mf, header, data, QUIET=quiet, MF=mf, TSY=tsy, NFP=nfp, FIELD=field, AT=at, FORCE=force

  ; check if have magnetic field in structure all ready

  IF NOT KEYWORD_SET(force) THEN BEGIN

    mf_str = ies_mag_bsc(header,data)

    IF KEYWORD_SET(mf_str) THEN BEGIN

      MESSAGE, 'WARNING, USING IF DATA MF TAG FOR MAGNETIC FIELD', /INFORMATIONAL

      RETURN, mf_str

    ENDIF

  ENDIF

  ; decide whether POLAR or CLUSTER

  IF STRPOS(header.datatype,'POLAR') EQ 0 THEN BEGIN

    RETURN, get_mf ( (header.tstart+header.tstop)/2.0, QUIET=quiet, MF=mf, TSY=tsy, NFP=nfp, FIELD=field )

  ENDIF ELSE IF STRPOS(header.datatype,'RAPID') EQ 0 THEN BEGIN

    RETURN, get_rapid_mf(STRMID(header.datatype,6,1),(header.tstart+header.tstop)/2.0, AT=at, QUIET=quiet, DATA=data)

  ENDIF ELSE BEGIN

    RETURN, 0

  ENDELSE

END


