;+
; Project     :	POLAR - CEPPAD
;
; Name        :	lshell_plot
;
; Purpose     :	Plots ALT, LSHELL, MLT, MLAT axes.
;               
; Explanation : Plots ALT, LSHELL, MLT, MLAT axes.      
;
; Use         : < lshell_plot, time_start, results, logoid >
;
; Inputs      : time_start       : 
;
; Opt. Inputs : logoid.
;
; Outputs     : 
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	eph_polar_spec_ax, anytim2utc, utc2tai.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	Adapted from cammice_survey.pro.
;
; Written     :	Version 0.0, Martin Carter, RAL, 22/10/96
;
; Modified    :	Version 0.1, 26/11/96
;                 Added conversion of LAT from km to Re.
;               Version 0.2, 12/5/97
;                 Changed some comments about start time.
;               Version 0.3, 5/10/98
;                 Changed so that expects cdf ephemeris structure.
;               Version 0.4, 22/07/02
;                 Added check that any ephemeris data in time range.
;
; Version     :	Version 0.4, 22/07/02
;-
;**********************************************************

PRO lshell_plot, time_start, results, logoid

  IF N_ELEMENTS(logoid) EQ 0 THEN logoid = 0L

  ; check if ephemeris data available

  IF N_TAGS(results) EQ 0 THEN RETURN

  ; save plot window and axis range

  plot_window = !P.POSITION  
  axis_crange = !X.CRANGE

  ; get character sizes in normal coords

  x_chsize = get_chsize( /X )
  y_chsize = get_chsize( /Y )

 ; divide region into number of axes

 y_space = (!P.REGION(3) - !P.REGION(1))/4.0

 yticklen = 0.0125

 taiday = utc2tai(anytim2utc(anytim2cal(results[0].time,/DATE)))

 sut = (results.time - taiday)/3600.
 alt = results.r/6378.
 lsh = results.l
 mlt = results.edmlt
 mlat = results.mlat

 if ( total ( alt ) le 0.0 ) then RETURN

 ; sut is time of ephemeris points in decimal hours of day
 ; current plot x axis is in secs from header start time (which will usually be time of first data point)
 ; !x.crange gives (possibly adjusted) time range of plot axis in secs from header start time i.e [-59.,121]
 ; want to convert the current plot range to decimal hours
 ; results.date contains the calendar time of the start of the day for the ephemeris data
 ; time_start contains TAI time of header start time

 tlims = (!X.CRANGE + time_start - taiday)/3600.0

 ; check any data within time limits

 IF sut[0] GT tlims[1] OR last_item(sut) LT tlims[0] THEN RETURN

 !P.POSITION(1) = !P.REGION(1)+3*y_space + y_chsize
 !P.POSITION(3) = !P.REGION(1)+4*y_space

 plot, tlims, [ 0.0, (!P.POSITION(3)-!P.POSITION(1))/y_chsize ], xstyle = 5, ystyle = 13, /noerase, xticklen = 0.0001, /nodata

 PLOTS, [!X.WINDOW(0), !X.WINDOW(1)], [!Y.WINDOW(0), !Y.WINDOW(0)], /NORMAL

 ; put axis title to left of axis

 xtitle = 'ALT:'
 XYOUTS, !P.POSITION(0) - x_chsize*(STRLEN(xtitle) + 0.5), !P.POSITION(1)-0.5*y_chsize, xtitle, /NORMAL, CHARSIZE=!P.CHARSIZE

 eph_polar_spec_ax, sut, alt, 0, tlims

 ; test cancel to bypass lshell plot

 res=WIDGET_EVENT(logoid, BAD_ID=bad_id, /NOWAIT)

 IF res.id NE 0 THEN BEGIN

   res=WIDGET_EVENT(logoid, BAD_ID=bad_id, /NOWAIT) ; swallow button release

   RETURN

 ENDIF

 !P.POSITION(1) = !P.REGION(1)+2*y_space + y_chsize
 !P.POSITION(3) = !P.REGION(1)+3*y_space

 plot, tlims, [ 0.0,(!P.POSITION(3)-!P.POSITION(1))/y_chsize ], xstyle = 5, ystyle = 13,          $
       /noerase, xticklen = 0.0001, YTICKLEN=yticklen, /nodata

 PLOTS, [!X.WINDOW(0), !X.WINDOW(1)], [!Y.WINDOW(0), !Y.WINDOW(0)], /NORMAL

 ; put axis title to left of axis

 xtitle = 'L-SH:'
 XYOUTS, !P.POSITION(0) - x_chsize*(STRLEN(xtitle) + 0.5), !P.POSITION(1)-0.5*y_chsize, xtitle, /NORMAL, CHARSIZE=!P.CHARSIZE

 eph_polar_spec_ax, sut, lsh, 2, tlims

 ; test cancel to bypass lshell plot

 res=WIDGET_EVENT(logoid, BAD_ID=bad_id, /NOWAIT)

 IF res.id NE 0 THEN BEGIN

   res=WIDGET_EVENT(logoid, BAD_ID=bad_id, /NOWAIT) ; swallow button release

   RETURN

 ENDIF

 !P.POSITION(1) = !P.REGION(1)+1*y_space + y_chsize
 !P.POSITION(3) = !P.REGION(1)+2*y_space

 plot, tlims, [ 0, (!P.POSITION(3)-!P.POSITION(1))/y_chsize ], xstyle = 5, ystyle = 13,          $
       /noerase, xticklen = 0.0001, YTICKLEN=yticklen, /nodata

 PLOTS, [!X.WINDOW(0), !X.WINDOW(1)], [!Y.WINDOW(0), !Y.WINDOW(0)], /NORMAL

 ; put axis title to left of axis

 xtitle = 'MLT:'
 XYOUTS, !P.POSITION(0) - x_chsize*(STRLEN(xtitle) + 0.5), !P.POSITION(1)-0.5*y_chsize, xtitle, /NORMAL, CHARSIZE=!P.CHARSIZE

 eph_polar_spec_ax, sut, mlt, 3, tlims

 ; test cancel to bypass lshell plot

 res=WIDGET_EVENT(logoid, BAD_ID=bad_id, /NOWAIT)

 IF res.id NE 0 THEN BEGIN

   res=WIDGET_EVENT(logoid, BAD_ID=bad_id, /NOWAIT) ; swallow button release

   RETURN

 ENDIF

 !P.POSITION(1) = !P.REGION(1)+0*y_space + y_chsize
 !P.POSITION(3) = !P.REGION(1)+1*y_space

 plot, tlims, [ 0.0, (!P.POSITION(3)-!P.POSITION(1))/y_chsize], xstyle = 5, ystyle = 13,          $
       /noerase, xticklen = 0.0001, YTICKLEN=yticklen, /nodata

 PLOTS, [!X.WINDOW(0), !X.WINDOW(1)], [!Y.WINDOW(0), !Y.WINDOW(0)], /NORMAL

 ; put axis title to left of axis

 xtitle = 'M LAT:'
 XYOUTS, !P.POSITION(0) - x_chsize*(STRLEN(xtitle) + 0.5), !P.POSITION(1)-0.5*y_chsize, xtitle, /NORMAL, CHARSIZE=!P.CHARSIZE

 eph_polar_spec_ax, sut, mlat, 1, tlims

 ; restore plot window

 !P.POSITION = plot_window

 !X.CRANGE   = axis_crange

END

