; read_cdf_at.pro
; Function to extract attitude information from the CDF orbit
; file supplied on the raw data or KP CD-ROMs.
;
; Arguments: y,		year (YYYY)
;	     m,		month (MM)
;	     d,		day (DD)
; Keywords:  CNT	Returns the number of records found
;            FILE	Specifies the location of the orbit file
;			otherwise the program will try to find an
;		        appropriate file.
;
; If the file cannot be found or does not cover the requested
; day then the function will return a value of -1 and the CNT
; keyword will be set to 0.
; On success the function returns a structure with the following
; specification and the CNT keyword is set to the number of records
;
; 	date 		YYYY-MM-DD
;	nrec		Number of records
;	sc		Spacecraft Name (PO for Polar)
;	time(nrec)	TAI time
;	gse_yaw
;	gse_pitch
;	gse_roll
;
;							CHP 11/96
;  Modified 19/2/97, MKC
;    Removed DEF type check and returned type via KEYWORD.
;  Modified 21/8/97, MKC
;    Handled case where epoch etc have only 1 element.
;       27/01/98 BEING MODIFIED TO COLLECT  AT DATA....
;  Modified 10/2/99, MKC
;    Added count=1 to ra and dec


FUNCTION read_cdf_at, file, CNT=nrec, TYPE=type

	nrec = 0				; Clear number of records
	; *** Open the CDF file ***
	IF ( N_ELEMENTS(file) NE 1) THEN   $
		file = STRING(y-1900,m,d,FORMAT="(I2.2,I2.2,I2.2,'01.cdf')")
	file1 = FINDFILE(file, COUNT=n)		; Check file path
	IF ( n NE 1 ) THEN RETURN, -1		; Invalid file specification
	id   = CDF_OPEN(file);

	; *** Get file information ***
	cdf = CDF_INQUIRE(id)

        ; *** Make sure we have the right orbit file ***
	CDF_ATTGET, id, "Logical_file_id", 0, lfid
	lfid = STR_SEP(lfid,"_")
        type = lfid(2)

;	IF ( N_ELEMENTS(lfid) NE 5 OR lfid(0) NE "PO" OR $
;		lfid(1) NE "OR" OR lfid(3) NE date ) THEN BEGIN
;	   	CDF_CLOSE, id
;		PRINT, "ERROR - Bad orbit file selected"
;   		RETURN,-1
;	ENDIF

	; *** Get the data that we're interested in **

   nrec = cdf.maxrec+1

   CDF_VARGET,id,"Epoch", REC_COUNT=nrec, COUNT=[1], epoch
   CDF_VARGET,id,"GSE_R_ASCENSION", REC_COUNT=nrec, COUNT=[1], RA
   CDF_VARGET,id,"GSE_DECLINATION", REC_COUNT=nrec, COUNT=[1], DEC

   CDF_CLOSE, id

   struct = REPLICATE({AT_STRUCT_V1, TIME:0.d0, GSE_RA: 0.0, GSE_DEC:0.0}, nrec)

	; *** For scalers get rid of the unused dimension ***
   epoch = REFORM(epoch)

   ; ra and dec in radians

   struct.gse_ra  = REFORM(ra)
   struct.gse_dec = REFORM(dec)

	; *** Convert Epoch to decimal hours on day ***
  FOR i=0,nrec-1 DO BEGIN
   CDF_EPOCH, epoch(i), year, month, day, hour, min, sec, milli, /break
   struct[i].time = hour*3600.0 + min*60.0 + sec + milli/1000.
  ENDFOR

  struct.time = struct.time + utc2tai(anytim2utc(STRING(year,month,day,FORMAT="(I4.4,'-',I2.2,'-',I2.2)" )))

  RETURN, struct

end
