; Function to extract ephemeris information from the CDF orbit
; file supplied on the raw data or KP CD-ROMs.
;
; Arguments: y,		year (YYYY)
;	     m,		month (MM)
;	     d,		day (DD)
; Keywords:  CNT	Returns the number of records found
;            FILE	Specifies the location of the orbit file
;			otherwise the program will try to find an
;		        appropriate file.
;
; If the file cannot be found or does not cover the requested
; day then the function will return a value of -1 and the CNT
; keyword will be set to 0.
; On success the function returns a structure with the following
; specification and the CNT keyword is set to the number of records
;
;	time(nrec)	Decimal UT hours
;	gse_xyz(3,nrec)	Position in GSE
;	r(nrec)		Radial distance
;	edmlt(nrec) 	ED Magnetic Local Time
;	mlat(nrec)	Magnetic latitude
;	l(nrec)		L Shell
;
;							CHP 11/96
;  Modified 19/2/97, MKC
;    Removed DEF type check and returned type via KEYWORD.
;  Modified 21/8/97, MKC
;    Handled case where epoch etc have only 1 element.
;  NB Dont really need year month day arguments if file supplied.
;  Modified 19/01/04, MKC
;    Added some printout if bad file ID

FUNCTION read_cdf_ephem, y, m, d, CNT=nrec, FILE=file, TYPE=type

	nrec = 0				; Clear number of records
	; *** Open the CDF file ***
	IF ( N_ELEMENTS(file) NE 1) THEN   $
		file = STRING(y-1900,m,d,FORMAT="(I2.2,I2.2,I2.2,'01.cdf')")
	file1 = FINDFILE(file, COUNT=n)		; Check file path
	IF ( n NE 1 ) THEN RETURN, -1		; Invalid file specification
	id   = CDF_OPEN( file );

	date = STRING(y,m,d,FORMAT="(I4.4,I2.2,I2.2)")

	; *** Get file information ***
	cdf = CDF_INQUIRE(id)

	; *** Make sure we have the right orbit file ***
	CDF_ATTGET, id, "Logical_file_id", 0, lfid
	lfid = STR_SEP(lfid,"_")
        type = lfid(2)
	IF ( N_ELEMENTS(lfid) NE 5 OR lfid(0) NE "PO" OR $
		lfid(1) NE "OR" OR lfid(3) NE date ) THEN BEGIN
	   	CDF_CLOSE, id
   		PRINT, "ERROR - Bad orbit file selected"
   		PRINT, "Logical file ID = ",lfid
   		RETURN,-1
	ENDIF

	; *** Get the data that we're interested in ***
	nrec = cdf.maxrec+1
	CDF_VARGET,id,"Epoch", REC_COUNT=nrec, COUNT=[1], epoch
	CDF_VARGET,id,"GSE_POS", REC_COUNT=nrec, pos
	CDF_VARGET,id,"EDMLT_TIME", REC_COUNT=nrec, COUNT=[1], mlt
	CDF_VARGET,id,"MAG_LATITUDE", REC_COUNT=nrec, COUNT=[1], mlat
	CDF_VARGET,id,"L_SHELL", REC_COUNT=nrec, COUNT=[1], lshell
	CDF_CLOSE, id

	; *** For scalers get rid of the unused dimension ***
	epoch = REFORM([epoch])

  struct = REPLICATE({OR_STRUCT_V1, TIME:0.d0, GSE_XYZ:FLTARR(3), R:0.0, EDMLT:0.0, MLAT:0.0, L:0.0}, nrec)

  struct.edmlt = REFORM([mlt])
  struct.mlat  = REFORM([mlat])
  struct.l     = REFORM([lshell])
  struct.gse_xyz = REFORM(pos)
  struct.r     = REFORM(SQRT( pos(0,*)^2 + pos(1,*)^2 + pos(2,*)^2))

	; *** Convert Epoch to decimal hours on day ***
	FOR i=0,nrec-1 DO BEGIN
		CDF_EPOCH, epoch(i), year, month, day, hour, min, sec, milli, /break
                struct[i].time = hour*3600.0 + min*60.0 + sec + milli/1000.
	ENDFOR

  struct.time = struct.time + utc2tai(anytim2utc(STRING(year,month,day,FORMAT="(I4.4,'-',I2.2,'-',I2.2)" )))

  RETURN, struct

END
