; Function to extract ephemeris information from the CDF MFE
; file supplied via CDAWEB.
;
; file :          	Specifies the location of the orbit file
;			otherwise the program will try to find an
;		        appropriate file.
;
; If the file cannot be found or does not cover the requested
; day then the function will return a value of 0.
;
; On success the function returns a structure with the following
; specification and the CNT keyword is set to the number of records
;
; 	date 		YYYY-MM-DD
;	time(nrec)	Decimal UT hours
;
;  MFE CDF variable names
;
;  Epoch  Epoch6  Time_PB5  POS_GSE  POS_GSM  POS_SM  Rad_Dist  TILT  MF_Num  MF_Timing  BT  BSC
;  COMP_DELTA  TRANS_DELTA  BTAV  B_GSE  B_GSM B_SM  MBTIGRF  MBCIGRF_GSE  MBCIGRF_GSM  MBCIGRF_SM
;  MBTSY  MBCTSYGSE  MBCTSYGSM  MBCTSYSM  N_FOOT_PRINT  S_FOOT_PRINT  dt_var  DQF POST_GAP  label_time
;  unit_time  format_time  label_posGSE  label_posGSM  label_posSM  label_BSC  label_BGSE  label_BGSM  label_BSM
;  label_IGRFGSE  label_IGRFGSM  label_IGRFSM  label_TSYGSE  label_TSYGSM  label_TSYSM  N_label_FOOTPRINT
;  S_label_FOOTPRINT  cartesian two_dims
;
;  Returns measured magnetic field strength in nT in SC coords in structure {MFE_STRUCT_BSC_V2, TIME:0.d0, BT: 0.0, MF:[0.,0.,0.]}
;  else if keyword_set
;  /GSE : returns measured magnetic field strength in nT in GSE coords in structure {MFE_STRUCT_GSE_V2, TIME:0.d0, BT: 0.0, MF:[0.,0.,0.]}
;  /TSY : returns calculated magnetic field strength in nT in GSE coords in structure {MFE_STRUCT_BTSY_V2, TIME:0.d0, BT: 0.0, MF:[0.,0.,0.]}
;  /NFP : returns lat and long of magnetic footprint in structure {MFE_STRUCT_NFP_V3, TIME:0.d0, NFP:FLTARR(2)}
;
;  Modified 19/2/97, MKC
;    Removed DEF type check and returned type via KEYWORD.
;  Modified 21/8/97, MKC
;    Handled case where epoch etc have only 1 element.
;  NB Dont really need year month day arguments if file supplied.
;  Modified 06/10/03, MKC
;    Changed structure name from MFE_STRUCT_BSC_V2 to MFE_STRUCT_BSC_XYZ_V3
;
FUNCTION read_cdf_mfe, file, TYPE=type, GSE=gse, TSY=tsy, NFP=nfp

	nrec = 0				; Clear number of records
	; *** Open the CDF file ***
	file1 = FINDFILE(file, COUNT=n)		; Check file path
	IF ( n NE 1 ) THEN RETURN, 0		; Invalid file specification
	id   = CDF_OPEN( file );

	; *** Get file information ***
	cdf = CDF_INQUIRE(id)

	; *** Make sure we have the right orbit file ***
	CDF_ATTGET, id, "Logical_file_id", 0, lfid
	lfid = STR_SEP(lfid,"_")
	IF ( N_ELEMENTS(lfid) NE 5 OR lfid(0) NE "po" OR $
		lfid(1) NE "k0" OR lfid(2) NE 'mfe' ) THEN BEGIN
	   	CDF_CLOSE, id
   		PRINT, "ERROR - Bad MFE file selected"
   		RETURN,0
	ENDIF

	; *** Get the data that we're interested in ***
        ; use for k = 0, 100 do help, CDF_VARINQ(id,k,/ZVARIABLE),/str to get variable names

        ; Epoch has 1563 values , MBTSY, B_GSE
        ; Time_PB5 has 1563 values
        ; Epoch6 has 14346 values, BSC

        IF KEYWORD_SET(gse) THEN BEGIN

          CDF_CONTROL, id, VARIABLE="Epoch", GET_VAR_INFO=v, SET_PADVALUE=0
          CDF_VARGET, id, "Epoch", REC_COUNT=v.maxrec+1, epoch
          CDF_VARGET, id, "B_GSE", REC_COUNT=v.maxrec+1, mf

    	  struct = {MFE_STRUCT_GSE_V2, TIME:0.d0, BT: 0.0, MF:[0.,0.,0.]}

        ENDIF ELSE IF KEYWORD_SET(tsy) THEN BEGIN

          CDF_CONTROL, id, VARIABLE="Epoch", GET_VAR_INFO=v, SET_PADVALUE=0
          CDF_VARGET, id, "Epoch", REC_COUNT=v.maxrec+1, epoch
          CDF_VARGET, id, "MBCTSYGSE", REC_COUNT=v.maxrec+1, mf

    	  struct = {MFE_STRUCT_BTSY_V2, TIME:0.d0, BT: 0.0, MF:[0.,0.,0.]}

        ENDIF ELSE IF KEYWORD_SET(nfp) THEN BEGIN

          CDF_CONTROL, id, VARIABLE="Epoch", GET_VAR_INFO=v, SET_PADVALUE=0
          CDF_VARGET, id, "Epoch", REC_COUNT=v.maxrec+1, epoch
          CDF_VARGET, id, "N_FOOT_PRINT", REC_COUNT=v.maxrec+1, mf

    	  struct = {MFE_STRUCT_NFP_V3, TIME:0.d0, NFP:FLTARR(2)}

        ENDIF ELSE BEGIN

          CDF_CONTROL, id, VARIABLE="Epoch6", GET_VAR_INFO=v, SET_PADVALUE=0
          CDF_VARGET, id, "Epoch6", REC_COUNT=v.maxrec+1, epoch
          CDF_VARGET, id, "BSC", REC_COUNT=v.maxrec+1, mf

    	  struct = {MFE_STRUCT_BSC_V2, TIME:0.d0, BT: 0.0, MF:[0.,0.,0.]}

        ENDELSE

	CDF_CLOSE, id

        ; clean up MF

        list = WHERE ( mf(0,*) GT -1e10, count)

        IF count EQ 0 THEN BEGIN
          PRINT, "ERROR - No good data"
          RETURN, 0
        ENDIF

        struct = REPLICATE(struct, count)

        IF KEYWORD_SET(nfp) THEN BEGIN

          struct.nfp = TEMPORARY(mf[*,list])

        ENDIF ELSE BEGIN

          struct.mf = TEMPORARY(mf[*,list])
          struct.bt = REFORM(SQRT(struct.mf[0]^2+struct.mf[1]^2+struct.mf[2]^2))

        ENDELSE

        ; *** For scalers get rid of the unused dimension ***
        ; NB did check that epoch here is [1,*] even without COUNT=1 specified

        epoch = REFORM([epoch])

        epoch = epoch[list]

        ; *** Convert Epoch to decimal hours on day ***
        FOR i=0,count-1 DO BEGIN
          CDF_EPOCH, epoch[i], year, month, day, hour, min, sec, milli, /break
          struct[i].time = hour*3600.0 + min*60.0 + sec + milli/1000.
	ENDFOR

        struct.time = struct.time + utc2tai(anytim2utc(STRING(year,month,day,FORMAT="(I4.4,'-',I2.2,'-',I2.2)" )))

	RETURN, struct

END
