;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_DO_CLEAN
;
; Purpose     :	Applies corrections to IES data.
;
; Explanation :	Corrects for pedestal shift.
;               This routine may be called for each sample point.
;               It uses a COMMON block to store information needed for each call.
;
; Use         : < rcs_do_clean, sc, times, lut_nos, array, ch_positions, BM=bm >
;
; Inputs      : sc           : STRING character giving spacecraft ID
;               times        : DBLARR(npoints) TAI time of EDB
;               lut_list     : STRUCTURE ARRAY containing LUT run info.
;               array        : FLTARR(nsectors,ndetectors,nbands,npoints) counts
;               ch_positions : INTARR(2,nbands,ndetectors,nluts) old energy channel positions in bins for each detector and LUT
;               new_ch_positions : INTARR(2,newbands,ndetectors,nluts) new energy channel positions in bins for each detector and LUT
;
; Opt. Inputs : None.
;
; Outputs     : array            : FLTARR(nsectors,ndetectors,newbands,npoints) counts
;
; Opt. Outputs:	None.
;
; Keywords    : BM        : FLAG indicates BM data
;               PEDESTALS : FLTARR(5,*) pedestal info array
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 6/2/01
;                 Added PEDESTALS keyword.
;               Version 0.2, 18/07/01
;                 Changed BM/NM flag to ies_shift.
;                 Cut out pedestal channels from interpolation.
;                 NM pedestal channels are not contiguous with signal channels ie
;                 channel 5 omitted from normal mode and does not make much sense to include pedestal channels
;                 anyway since can only give pedestal contamination.
;                 Previously, where start of first signal channel beyond start of first resampled energy channel
;                 then would interpolate using pedestal channel giving pedestal contamination.
;                 Removing pedestal channels from interpolation then have the problem how to extrapolate where
;                 necessary from signal channels. Existing code simply truncates at first signal channel and effectively
;                 extrapolates using zeroes outside of this.
;               Version 0.3, 15/08/01
;                 Added FLAGS keyword.
;               Version 0.4, 02/12/02
;                 Have modified ies interp so that now allows extrapolation outside of actual energy range.
;               Version 0.5, 24/10/03
;                 Used ies lut list.
;               Version 0.6, 20/11/03
;                 Used ch_positions [bins] instead of output_chans[bins].
;                 Renamed rcs clean to rcs do clean.
;                 Used routine rcs get calibration and changed argument list of rcs pedestals.
;                 Moved rcs get pedestal calibration to rcs pedestals.
;               Version 0.7, 01/12/03
;                 Fixed bug where lut_no not adjusted.
;               Version 0.8, 17/12/03
;                 Added ch_positions to argument list.
;                 Changed new_ch_positions to an input.
;                 Replaced lut_nos input with lut_list.
;
; Version     :	Version 0.8, 17/12/03
;-
;**********************************************************

PRO rcs_do_clean, sc, times, lut_list, array, ch_positions, new_ch_positions, BM=bm, PEDESTALS=pedestals, FLAGS=flags

  npoints = N_ELEMENTS(times)

  no_of_detectors = (SIZE(ch_positions))[3]

  nbands = (SIZE(ch_positions))[2]

  new_bands = (SIZE(new_ch_positions))[2]

  IF KEYWORD_SET(bm) THEN BEGIN

    npeds = 4

    no_of_sectors = 16

  ENDIF ELSE BEGIN

    npeds = 2

    no_of_sectors = 1

  ENDELSE

  ; array        : FLTARR(nsectors,ndetectors,nbands,npoints)

  array = REFORM(array, no_of_sectors, no_of_detectors, nbands, npoints)

  temp_array = REFORM(TRANSPOSE(array, [2,1,0,3]), nbands, no_of_detectors, no_of_sectors*npoints)

  ; get pedestal info
  ; Inputs      : npeds         : INT no. of pedestal channels
  ;               temp_array    : FLTARR( nbands, ndetectors, nsectors*npoints) containing input data
  ;               times         : DBLARR(npoints) TAI times of samples
  ;               lut_list      : STRUCTURE ARRAY containing LUT run info.
  ;               nsectors      : INT no. of sectors.
  ;               detector_list : INTARR(ndetectors) detector numbers used 0-8
  ; Outputs     : pedestals : FLTARR ( 4, ndetectors, npoints*nsectors)
  ;                           0 : pedestal shift from central bin for cross_correlation algorithm
  ;                           1 : pedestal shift from central bin for centroid algorithm
  ;                           2 : occupancy defined as the ratio of non-pedestal counts to measured total counts
  ;                           3 : visible counts defined as ratio of counts in data channels vs expected total counts

  pedestals = rcs_pedestals ( sc, npeds, temp_array, times, lut_list, no_of_sectors, INDGEN(no_of_detectors), BM=bm)

  ; do resampling
  ; Inputs      : lut_list      : STRUCTURE ARRAY containing LUT run info.
  ;               nsectors      : INT no. of sectors.
  ;               new_bands     : INT new no. of energies.
  ;               ndetectors    : INT no. of detectors.
  ;               nbands        : INT no. of energy channels.
  ;               ch_positions  : INTARR(2,nbands,ndetectors,nluts) containing energy channel positions in bins for each detector and LUT
  ;               shifted_pedestals : FLTARR ( nsamples, nsectors, ndetectors) pedestal bin positions
  ;               array         : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
  ; Outputs     : new_bands     : INT new no. of energy channels
  ;               ch_positions  : INTARR(2,new_bands,ndetectors,nluts) new energy channel positions in bins for each detector and LUT
  ;               array         : FLTARR(nsectors, ndetectors, new_bands, npoints) rebinned data.

  shifted_pedestals = TRANSPOSE(REFORM(pedestals[0,*,*],no_of_detectors,no_of_sectors,npoints),[2,1,0])

  ; omit pedestal channels from interpolation

  array = array[*,*,npeds:*,*]

  ; ch_positions  : INTARR(2,nbands,ndetectors,nluts) containing energy channel bin positions for each detector and LUT

  old_ch_positions = ch_positions[*,npeds:*,*,*]

  nbands = nbands - npeds

  ; set up ch_positions in bins
  ; NB uses POLAR energy channels

  ies_shift, lut_list, no_of_sectors, new_bands, no_of_detectors, nbands, old_ch_positions, new_ch_positions, $
             shifted_pedestals, array, FLAGS=flags

END
