;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_DO_E3DD
;
; Purpose     :	Processes E3DD data.
;
; Explanation :	Processes E3DD data to produce various IF data sets.
;
; Use         : < rcs_do_e3dd, use, e3dd, e3ddbm, mvals, lut_no, flag, sc, vs, time>
;
; Inputs      : use        : STRUCTURE containing flags indicating which IF data
;               e3dd       : FLTARR(8,9) raw counts for NM energy channels
;               e3ddbm     : FLTARR(16,9,12) raw BM counts
;               mvals      : BYTARR(11) giving mtable, msigns and epad table
;               lut_no     : INT IES LUT 0:3
;               flag       : BYTE flag bits
;               sc         : STRING character giving spacecraft ID
;               vs         : STRING IF version 'sss_v_a_b'
;               time       : DOUBLE TAI time of EDB
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    :
;
; Modified    :	Version 0.1, 6/2/01
;                 Added pedestal and espct IF data set.
;               Version 0.2, 08/04/01
;                 Used rcs espct.
;               Version 0.3, 18/07/01
;                 Added CDROM version number.
;               Version 0.4, 15/08/01
;                 Added FLAGS to rcs clean.
;               Version 0.5, 08/02/02
;                 Added EPAD2.
;               Version 0.6, 28/03/02
;                 Added check on E3DD counts.
;               Version 0.7, 12/04/02
;                 Forced check on E3DD if processing EPAD.
;               Version 0.8, 04/07/02
;                 Split into new routine.
;               Version 0.9, 22/07/02
;                 Allowed possibility that epad not defined.
;               Version 1.0, 02/12/02
;                 Passed flag into rcs clean for BM.
;                 Previously only passed on for NM.
;               Version 1.1, 20/11/03
;                 Renamed rcs clean to rcs do clean.
;                 Renamed rcs e3dd to rcs do e3dd.
;               Version 1.2, 17/12/03
;                 Used ies set ch positions.
;                 Changed output_chans to ch_positions.
;                 Moved adjustment of lut_no and set up of ch_positions to this routine.
;                 Changed rcs espct argument list.
;                 Changed rcs espct to rcs do espct.
;
; Version     :	Version 1.2, 17/12/03
;-
;**********************************************************

PRO rcs_do_e3dd, use, e3dd, e3ddbm, mvals, lut_no, flag, sc, vs, time

  IF KEYWORD_SET(e3ddbm) THEN BEGIN

    ; add to e3ddbm common block
    ; e3ddbm = FLTARR ( nsectors, ndetectors, nenergies )

    IF KEYWORD_SET(use.e3ddbm) AND KEYWORD_SET(e3ddbm) THEN rcs_add_e3ddbm, sc, vs, time, e3ddbm, flag, lut_no

    IF KEYWORD_SET(use.cleanbm) OR KEYWORD_SET(use.cleanbm2) THEN BEGIN

      ; e3ddbm = [sectors, detectors, energies]

      cleanbm = e3ddbm

      ; get energy channels

      ch_positions = rcs_get_luts ( sc, 'E3DDBM', lut_times)

      new_ch_positions = ies_set_ch_positions(8, 9, N_ELEMENTS(ch_positions[0,0,0,*]), RAPID=2 )

      ; adjust lut_no for changes

      lut = rcs_adjust_lut (time, lut_no, lut_times)

      ; form lut run structure

      lut_list = ies_lut_list(lut)

      ; clean data

      rcs_do_clean, sc, time, lut_list, cleanbm, ch_positions, new_ch_positions, /BM, FLAGS=flag

      ; clean data

      ;rcs_do_clean_sample, sc, time, lut, cleanbm, ch_positions, new_ch_positions, /BM, FLAGS=flag

      ; cleanbm is FLTARR(nsectors,ndetectors,newbands)

      IF KEYWORD_SET(use.cleanbm) THEN rcs_add_cleanbm, sc, vs, time, cleanbm, mvals, flag, lut_no, new_ch_positions

      IF KEYWORD_SET(use.cleanbm2) THEN BEGIN

        ; sum over energies -- including the lowest channel
        ; e3ddbm is FLTARR(nsectors,ndetectors)

        cleanbm = TOTAL(cleanbm[*,*,1:*],3)
        ; cleanbm = TOTAL(cleanbm,3)

        dims = SIZE(new_ch_positions)

        new_ch_positions = REFORM(new_ch_positions,dims[1]*dims[2],dims[3],dims[4])

        new_ch_positions = new_ch_positions[[2,2*dims[2]-1],*,*]
        ; new_ch_positions = new_ch_positions[[0,2*dims[2]-1],*,*]

        new_ch_positions = REFORM(new_ch_positions,2,1,dims[3],dims[4])

        rcs_add_cleanbm2, sc, vs, time, cleanbm, mvals, flag, lut_no, new_ch_positions

      ENDIF

    ENDIF

  ENDIF

  IF KEYWORD_SET(use.e3dd)OR KEYWORD_SET(use.clean) OR $
     KEYWORD_SET(use.pedestal) OR KEYWORD_SET(use.espct) THEN BEGIN

    IF KEYWORD_SET(e3ddbm) THEN BEGIN

      ; spin average [9,12]

      e3dd = TOTAL(e3ddbm,1)

      ; select NM channels
      ; and combine last 2 sets of channels

      e3dd[*,0:5] = e3dd[*,[1,2,4,5,6,7]]

      e3dd[*,6] = e3dd[*,8] + e3dd[*,9]

      e3dd[*,7] = e3dd[*,10] + e3dd[*,11]

      e3dd = e3dd[*,0:7]

    ENDIF ELSE BEGIN

      ; convert e3dd to LONARR[9,8]

      e3dd = TRANSPOSE(e3dd)

    ENDELSE

    ; check E3DD counts

    dummy = rcs_check_ies(lut_no, e3dd, flag, /CHECK)

    ; e3dd = FLTARR ( ndetectors, nenergies )

    IF KEYWORD_SET(use.e3dd) THEN rcs_add_e3dd, sc, vs, time, e3dd, flag, lut_no

    IF KEYWORD_SET(use.clean) OR KEYWORD_SET(use.pedestal) OR KEYWORD_SET(use.espct) THEN BEGIN

      ; get energy channels

      ch_positions = rcs_get_luts ( sc, 'E3DD', lut_times)

      new_ch_positions = ies_set_ch_positions(6, 9, N_ELEMENTS(ch_positions[0,0,0,*]), RAPID=1 )

      ; adjust lut_no for changes

      lut = rcs_adjust_lut (time, lut_no, lut_times)

      ; form lut run structure

      lut_list = ies_lut_list(lut)

      ; clean data

      rcs_do_clean, sc, time, lut_list, e3dd, ch_positions, new_ch_positions, PEDESTALS=pedestals, FLAGS=flag

      ; clean data

      ;rcs_do_clean_sample, sc, time, lut, e3dd, ch_positions, new_ch_positions, PEDESTALS=pedestals, FLAGS=flag

    ENDIF

    ; e3dd is FLTARR(nsectors,ndetectors,newbands,npoints)

    IF KEYWORD_SET(use.clean) THEN rcs_add_clean, sc, vs, time, e3dd, flag, lut_no, new_ch_positions

    ; pedestals : FLTARR ( 5, ndetectors)
    ; convert to : FLTARR ( ndetectors, nenergies )

    IF KEYWORD_SET(use.pedestal) THEN rcs_add_pedestal, sc, vs, time, TRANSPOSE(pedestals), flag, lut_no

    IF KEYWORD_SET(use.espct) THEN BEGIN

      ; average over detectors

      rcs_do_espct, sc, e3dd

      ; e3dd is FLTARR(nsectors,newbands,npoints)

      rcs_add_espct, sc, vs, time, e3dd, flag, lut_no, new_ch_positions

    ENDIF

  ENDIF

END
