;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_EPAD
;
; Purpose     :	Processes EPAD data.
;
; Explanation :	Processes EPAD/E3DDBM data to produce various IF data sets.
;
; Use         : < rcs_epad, use, epad, e3ddbm, mvals, lut_no, flag, sc, vs, time>
;
; Inputs      : use        : STRUCTURE containing flags indicating which IF data
;               epad       : FLTARR(16,3,2) raw counts for EPAD energy channels
;               e3ddbm     : FLTARR(16,9,12) raw BM counts
;               mvals      : BYTARR(11) giving mtable, msigns and epad table
;               lut_no     : INT IES LUT 0:3
;               flag       : BYTE flag bits
;               sc         : STRING character giving spacecraft ID
;               vs         : STRING IF version 'sss_v_a_b'
;               time       : DOUBLE TAI time of EDB
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    :
;
; Modified    :	Version 0.1, 6/2/01
;                 Added pedestal and espct IF data set.
;               Version 0.2, 08/04/01
;                 Used rcs espct.
;               Version 0.3, 18/07/01
;                 Added CDROM version number.
;               Version 0.4, 15/08/01
;                 Added FLAGS to rcs clean.
;               Version 0.5, 08/02/02
;                 Added EPAD2.
;               Version 0.6, 28/03/02
;                 Added check on E3DD counts.
;               Version 0.7, 12/04/02
;                 Forced check on E3DD if processing EPAD.
;               Version 0.8, 04/07/02
;                 Split into new routine.
;               Version 0.9, 22/07/02
;                 Allowed possibility that epad not defined.
;
; Version     :	Version 0.9, 22/07/02
;-
;**********************************************************

PRO rcs_epad, use, epad, e3ddbm, mvals, lut_no, flag, sc, vs, time

  IF KEYWORD_SET(use.epad) OR KEYWORD_SET(use.epad2)  THEN BEGIN

    IF KEYWORD_SET(e3ddbm) THEN BEGIN

      ; get epad energies [16, 9, 2]

      epad_3d = TOTAL(REFORM(e3ddbm[*,*,4:11],16,9,4,2),3)

      IF KEYWORD_SET(use.epad) THEN BEGIN

        rcs_add_epad, sc, vs, time, epad_3d, mvals, flag, lut_no

      ENDIF

      IF KEYWORD_SET(use.epad2) THEN BEGIN

        ; extract epad_table

        epad_table = mvals[10]

        ; get detectors for each of the 16 sectors [3,16]

        tab = rcs_decode_m ( mvals)

        msigns = rcs_decode_msigns(mvals)

        mtab = rcs_decode_m ( mvals, /MVAL)

        ; loop through sectors
        ; extract 3 detectors used for each sector based on m value

        FOR s = 0, 15 DO rcs_add_epad2, sc, vs, time+s*0.25, epad_3d[s,tab[*,s],*], [mtab[s],msigns[s],FIX(epad_table)], flag, lut_no

      ENDIF

      ; if have epad from another source check they are the same

      IF 0 THEN BEGIN
      ; IF KEYWORD_SET(epad) THEN BEGIN

        ; get detectors for each of the 16 sectors [3,16]

        tab = rcs_decode_m ( mvals)

        ; rotate tab to [16,3]

        tab = transpose(tab)

        ; convert tab to index [sectors,detectors]

        tab = tab + INDGEN(16)#REPLICATE(1,3)*16

        ; combine sectors and detectors

        epad_3d = REFORM(epad_3d, 16*9, 2)

        ; index into array to give array [16,3,2]

        epad_3d = epad_3d[tab,*]

        ; work out abs differences

        diff = TOTAL(ABS(epad_3d-epad))

        ; check differences

        IF diff NE 0.0 THEN rcs_message, /IMPLEMENTATION

      ENDIF

    ENDIF ELSE BEGIN

      ; get detectors for each of the 16 sectors [3,16]

      tab = rcs_decode_m ( mvals)

      IF KEYWORD_SET(use.epad) THEN BEGIN

        ; set up 3D array [sectors, detectors, energies]

        epad_3d = FLTARR(16,9,2)

        ; loop through sectors
        ; select 3 detectors used for each sector based on m value

        FOR s = 0, 15 DO epad_3d[s,tab[*,s],*] = epad[s,*,*]

        rcs_add_epad, sc, vs, time, epad_3d, mvals, flag, lut_no

      ENDIF

      IF KEYWORD_SET(use.epad2) THEN BEGIN

        ; extract epad_table

        epad_table = mvals[10]

        msigns = rcs_decode_msigns(mvals)

        mtab = rcs_decode_m ( mvals, /MVAL)

        ; loop through sectors
        ; 3 detectors used for each sector based on m value

        FOR s = 0, 15 DO rcs_add_epad2, sc, vs, time+s*0.25, epad[s,*,*], [mtab[s],msigns[s],FIX(epad_table)], flag, lut_no

      ENDIF

    ENDELSE

  ENDIF

END
