;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_EXTRACT_BM
;
; Purpose     :	Extracts data from BM EDB and constructs IF data.
;
; Use         : < rcs_extract_bm, use, edb, time, sc, vs, flag, lut_no, $
;                     epad_table, fgm, hist_mode >
;
; Inputs      : use        : STRUCTURE containing flags indicating which IF data
;                            sets to make.
;               edb        : BYTARR EDB packet
;               time       : DOUBLE TAI time of EDB
;               sc         : STRING character giving spacecraft ID
;               vs         : STRING IF version 'sss_v_a_b'
;               flag       : BYTE flag bits
;               lut_no     : INT IES LUT 0:3 (0:8 in histogram mode)
;               epad_table : FLAG indicating epad table A (0) or B (1)
;               fgm        : FLAG indicating whether mag field available
;                            in all sectors
;               hist_mode  : FLAG indicating IES histogram mode active
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, 6/2/01
;                 Added pedestal and espct IF data set.
;               Version 0.2, 08/04/01
;                 Used rcs espct.
;               Version 0.3, 15/05/01
;                 Added CLEANBM2..
;               Version 0.4, 09/07/01
;                 Corrected bug in deriving last 2 channels in NM from BM.
;               Version 0.5, 18/07/01
;                 Added CDROM version number.
;               Version 0.6, 15/08/01
;                 Added FLAGS to rcs clean.
;               Version 0.7, 08/02/02
;                 Added EPAD2.
;               Version 0.8, 27/02/02
;                 Added mvals to cleanbm and cleanbm2.
;               Version 0.9, 28/03/02
;                 Added check on E3DD counts.
;               Version 1.0, 12/04/02
;                 Forced check on E3DD if processing EPAD.
;               Version 1.1, 23/07/02
;                 Split into new routines.
;               Version 1.2, 29/11/03
;                 Renamed rcs e3dd to rcs do e3dd.
;
; Version     :	Version 1.2, 29/11/03
;-
;**********************************************************

PRO rcs_extract_bm, use, edb, time, sc, vs, flag, lut_no, epad_table, fgm, hist_mode

  ; BM1

  ; generate dummy data

  ; rcs_test_bm, edb, hist_mode, lut_no

  ; check hist mode

  IF hist_mode THEN BEGIN

    rcs_extract_bmhist, use, edb, time, sc, vs, flag, lut_no

  ENDIF ELSE BEGIN

    IF KEYWORD_SET(use.e3ddbm) OR KEYWORD_SET(use.e3dd) OR $
       KEYWORD_SET(use.epad) OR KEYWORD_SET(use.epad2) OR $
       KEYWORD_SET(use.clean) OR KEYWORD_SET(use.cleanbm) OR $
       KEYWORD_SET(use.cleanbm2) OR KEYWORD_SET(use.pedestal) OR $
       KEYWORD_SET(use.espct) THEN BEGIN

      ; e3ddbm is LONARR[1152]  = 12 energies, 8 or 16 sectors, 9 detectors

      e3ddbm = rcs_decompress(edb['478'X:'478'X+1152-1])

      ; e3ddbm is LONARR[12,96] see DARD

      e3ddbm = REFORM(e3ddbm, 12, 96)

      ; resample at 16 sector resolution splitting counts between bins
      ; reform into [energies,detectors,sectors]

      start = e3ddbm[*,0:23]

      start = REFORM(start,12,3,8)

      start = REBIN(start,12,3,16,/SAMPLE)/2.

      middle = e3ddbm[*,24:71]

      middle = REFORM(middle,12,3,16)

      finish = e3ddbm[*,72:95]

      finish = REFORM(finish,12,3,8)

      finish = REBIN(finish,12,3,16,/SAMPLE)/2.

      ; create [12,9,16]

      e3ddbm = [[start],[middle],[finish]]

      ; rotate to [sectors, detectors, energies]

      e3ddbm = TRANSPOSE(e3ddbm,[2,1,0])

    ENDIF

    ; extract mvals

    IF KEYWORD_SET(use.epad) OR KEYWORD_SET(use.epad2) OR $
       KEYWORD_SET(use.cleanbm) OR KEYWORD_SET(use.cleanbm2) THEN BEGIN

      ; set up mvals array

      mvals = rcs_get_mvals(edb, epad_table, /BM)

    ENDIF

    ; NB lut_no has not been adjusted for LUT changes

    rcs_do_e3dd, use, 0, e3ddbm, mvals, lut_no, flag, sc, vs, time

    rcs_epad, use, 0, e3ddbm, mvals, lut_no, flag, sc, vs, time

  ENDELSE

  IF KEYWORD_SET(use.i3dd) THEN BEGIN

    ; extract I3DD [energies,detectors]

    i3dd = REFORM(rcs_decompress(edb['215'X:'215'X+576-1]),144,4)

    ; get subc

    subc = edb['003'X]

    ; send to routine in chunks as in NM, fake the subc

    FOR k = 0, 3 DO rcs_add_i3dd, sc, vs, time, i3dd[*,k], ((4*subc) MOD 32) + k, flag

  ENDIF

  IF KEYWORD_SET(use.ipad) THEN BEGIN

    ; extract IPAD [energies,detectors,sectors]

    ipad = REFORM(rcs_decompress(edb['1B2'X:'1B2'X+96-1]),2,3,16)

    ; set up mvals array

    IF NOT KEYWORD_SET(mvals) THEN mvals = rcs_get_mvals(edb, epad_table, /BM)

    rcs_ipad, ipad, time, sc, vs, mvals, flag

  ENDIF

  ; extract HSPCT [energies]

  IF KEYWORD_SET(use.hspct) THEN BEGIN

    hspct = REFORM(rcs_decompress(edb['148'X:'148'X+8-1]),8)

    rcs_add_hspct, sc, vs, time, hspct, flag

  ENDIF

  ; extract ISPCT [energies]

  IF KEYWORD_SET(use.ispct) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    ispct = REFORM(rcs_decompress(edb['150'X:'150'X+4-1]),4)

    rcs_add_ispct, sc, vs, time, ispct, subc, flag

  ENDIF

  ; extract MTRX

  IF KEYWORD_SET(use.mtrx) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    mtrx = REFORM(rcs_decompress(edb['455'X:'455'X+32-1]),8,4)

    ; send to routine in chunks as in NM, fake the subc

    FOR k = 0, 3 DO rcs_add_mtrx, sc, vs, time, mtrx[*,k], 4*subc + k, flag

  ENDIF

  ; extract SGLBM

  IF KEYWORD_SET(use.sglbm) THEN BEGIN

    sglbm = rcs_decompress(edb['156'X:'156'X+80-1])

    ; reform into detectors = [ENY,TCR,TAC,STA,STO] vs sectors

    sglbm = REFORM(sglbm,5,16)

    ; send to routine as [nsectors,ndetectors]

    rcs_add_sglbm, sc, vs, time, TRANSPOSE(sglbm), flag

  ENDIF

  ; extract SGL01

  IF KEYWORD_SET(use.sgl01) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    sgl01 = rcs_decompress(edb['156'X:'156'X+80-1])

    ; reform into detectors = [ENY,TCR,TAC,STA,STO] vs sectors

    sgl01 = REFORM(sgl01,5,16)

    ; send to routine in chunks as in NM, fake the subc
    ; sum over sectors as in NM

    rcs_add_sgl01, sc, vs, time, [TOTAL(sgl01[3,0:7]), TOTAL(sgl01[0,*])], 0, flag

    rcs_add_sgl01, sc, vs, time, [TOTAL(sgl01[3,8:15]),TOTAL(sgl01[1,*])], 1, flag

    rcs_add_sgl01, sc, vs, time, [TOTAL(sgl01[4,0:7]), TOTAL(sgl01[2,*])], 2, flag

    rcs_add_sgl01, sc, vs, time, [TOTAL(sgl01[4,8:15]),TOTAL(sgl01[3,*])], 3, flag

  ENDIF

  ; extract SGL2

  IF KEYWORD_SET(use.sgl2) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    sgl2 = REFORM(rcs_decompress(edb['1A6'X:'1A6'X+9-1]),3,3)

    ; send to routine in chunks as in NM, fake the subc

    FOR k = 0, 2 DO rcs_add_sgl2, sc, vs, time, sgl2[*,k], 3*(subc MOD 2) + k, flag

  ENDIF

  ; extract SGL3

  IF KEYWORD_SET(use.sgl3) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    sgl3 = REFORM(rcs_decompress(edb['1AF'X:'1AF'X+3-1]),3)

    FOR k = 0, 2 DO rcs_add_sgl3, sc, vs, time, sgl3[k], 3*(subc MOD 8) + k, flag

  ENDIF

END
