;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_EXTRACT_NM
;
; Purpose     :	Extracts data from NM EDB and constructs IF data.
;
; Use         : < rcs_extract_nm, use, edb, time, sc, vs, flag, lut_no, $
;                     epad_table, fgm, hist_mode >
;
; Inputs      : use        : STRUCTURE containing flags indicating which IF data
;                            sets to make.
;               edb        : BYTARR EDB packet
;               time       : DOUBLE TAI time of EDB
;               sc         : STRING character giving spacecraft ID
;               vs         : STRING IF version 'sss_v_a_b'
;               flag       : BYTE flag bits
;               lut_no     : INT IES LUT 0:3 (0:8 in histogram mode)
;               epad_table : BYTE FLAG indicating epad table A (0) or B (1)
;               fgm        : FLAG indicating whether mag field available
;                            in all sectors
;               hist_mode  : FLAG indicating IES histogram mode active
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Modified    :	Version 0.1, 6/2/01
;                 Added pedestal and espct IF data set.
;               Version 0.2, 08/04/01
;                 Used rcs espct.
;               Version 0.3, 18/07/01
;                 Added CDROM version number.
;               Version 0.4, 15/08/01
;                 Added FLAGS to rcs clean.
;               Version 0.5, 08/02/02
;                 Added EPAD2.
;               Version 0.6, 28/03/02
;                 Added check on E3DD counts.
;               Version 0.7, 12/04/02
;                 Forced check on E3DD if processing EPAD.
;               Version 0.8, 23/07/02
;                 Split into new routines.
;               Version 0.9, 29/11/03
;                 Renamed rcs e3dd to rcs do e3dd.
;
; Version     :	Version 0.9, 29/11/03
;-
;**********************************************************

PRO rcs_extract_nm, use, edb, time, sc, vs, flag, lut_no, epad_table, fgm, hist_mode

  ; generate test data

  ; rcs_test_nm, edb

  ; NM/BM2

  ; check hist mode

  IF hist_mode THEN BEGIN

    rcs_extract_nmhist, use, edb, time, sc, vs, flag, lut_no

  ENDIF ELSE BEGIN

    IF KEYWORD_SET(use.e3dd)OR KEYWORD_SET(use.clean) OR $
       KEYWORD_SET(use.pedestal) OR KEYWORD_SET(use.espct) OR $
       KEYWORD_SET(use.epad) OR KEYWORD_SET(use.epad2)  THEN BEGIN

      e3dd = REFORM(rcs_decompress(edb['1B0'X:'1B0'X+72-1]),8,9)

    ENDIF

    ; extract EPAD [energies,detectors,sectors]

    IF KEYWORD_SET(use.epad) OR KEYWORD_SET(use.epad2) THEN BEGIN

      epad = REFORM(rcs_decompress(edb['150'X:'150'X+96-1]),2,3,16)

      ; rotate to [sectors,detectors,energies]

      epad = TRANSPOSE(epad,[2,1,0])

      ; set up mvals array

      mvals = rcs_get_mvals(edb, epad_table)

    ENDIF

    ; NB lut_no has not been adjusted for LUT changes

    rcs_do_e3dd, use, e3dd, 0, mvals, lut_no, flag, sc, vs, time

    rcs_epad, use, epad, 0, mvals, lut_no, flag, sc, vs, time

  ENDELSE

  IF KEYWORD_SET(use.i3dd) THEN BEGIN

    ; extract I3DD

    i3dd = REFORM(rcs_decompress(edb['0B7'X:'0B7'X+144-1]),144)

    ; get subc

    subc = edb['003'X]

    rcs_add_i3dd, sc, vs, time, i3dd, subc, flag

  ENDIF

  IF KEYWORD_SET(use.ipad) THEN BEGIN

    ; extract IPAD [energies,detectors,sectors]

    ipad = REFORM(rcs_decompress(edb['057'X:'057'X+96-1]),2,3,16)

    ; set up mvals array

    IF NOT KEYWORD_SET(mvals) THEN mvals = rcs_get_mvals(edb, epad_table)

    rcs_ipad, ipad, time, sc, vs, mvals, flag

  ENDIF

  ; extract HSPCT [energies]

  IF KEYWORD_SET(use.hspct) THEN BEGIN

    hspct = REFORM(rcs_decompress(edb['046'X:'046'X+8-1]),8)

    rcs_add_hspct, sc, vs, time, hspct, flag

  ENDIF

  ; extract ISPCT [energies]

  IF KEYWORD_SET(use.ispct) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    ispct = REFORM(rcs_decompress(edb['04E'X:'04E'X+4-1]),4)

    rcs_add_ispct, sc, vs, time, ispct, subc, flag

  ENDIF

  ; extract MTRX

  IF KEYWORD_SET(use.mtrx) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    mtrx = REFORM(rcs_decompress(edb['147'X:'147'X+8-1]),8)

    rcs_add_mtrx, sc, vs, time, mtrx, subc, flag

  ENDIF

  ; extract SGL01

  IF KEYWORD_SET(use.sgl01) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    sgl0 = rcs_decompress(edb['009'X])

    sgl1 = rcs_decompress(edb['052'X])

    rcs_add_sgl01, sc, vs, time, [sgl0,sgl1], subc, flag

  ENDIF

  ; extract SGL2

  IF KEYWORD_SET(use.sgl2) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    sgl2 = REFORM(rcs_decompress(edb['053'X:'053'X+3-1]),3)

    rcs_add_sgl2, sc, vs, time, sgl2, subc, flag

  ENDIF

  ; extract SGL3

  IF KEYWORD_SET(use.sgl3) THEN BEGIN

    ; get subc

    subc = edb['003'X]

    sgl3 = rcs_decompress(edb['056'X])

    rcs_add_sgl3, sc, vs, time, sgl3, subc, flag

  ENDIF

END
