;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PEDESTAL_SAMPLES
;
; Purpose     : Gets pedestal sample info for given detector and LUT.
;
; Explanation :	Gets pedestal sample info for given detector and LUT.
;
; Use         : < rcs_pedestal_samples, sc, detector, lut, samples, max_samples_lists, pedestal_centre, channel_centres, $
;                                       pedestal_posn, BM=bm >
;
; Inputs      : sc       : STRING character SC
;               detector : INT detector no. 0-8
;               lut      : INT LUT no
;
; Opt. Inputs : None.
;
; Outputs     : samples          : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
;                                  data sampled as for the real data. Only the first npeds energy channels are kept.
;               max_sample_list  : FLTARR(1+nshifts, npeds) indices of shifts with peak in channel 1+channel.
;                                  Element 0 contains no of indices.max_samples contains the index of the peak
;                                  for each shift. This is used to restrict the search for max cross-correlation
;                                  to samples with the same peak position as the data
;               pedestal_centre  : FLOAT position of center of pedestal channels.
;               channel_centres  : FLTARR(npeds) position of centers of pedestal channels.
;               pedestal_posn    : FLOAT result of fit to pedestal.
;               pedestal_width   : FLOAT result of fit to pedestal (Gaussian width).
;
; Opt. Outputs:	None.
;
; Keywords    : BM           : FLAG indicating BM rather than NM.
;               FORCE        : FLAG indicating should force set up of calibration data and
;                                   calculation
;
; Written     :	Version 0.0, 21/11/03
;                 Separated from rcs pedestals.
;
; Modified    : Version 0.1, 12/12/03
;                 Added sc argument.
;               Version 0.2, 19/01/04
;                 Corrected bug : COMMON block not reset for new SC.
;               Version 0.4, 02/02/04
;                 Added FORCE keyword.
;                 Added pedestal_width to argument list
;
; Version     :	Version 0.4, 02/02/04
;-
;**********************************************************

PRO rcs_pedestal_samples, sc, detector, lut, samples, max_sample_lists, pedestal_centre, channel_centres, $
                          pedestal_posn, pedestal_width, BM=bm, FORCE=force

  COMMON nm_pedestal_sample_block, nm_info, nm_samples, nm_max_sample_lists, nm_pedestal_centres, nm_channel_centres, nm_pedestal_posns, nm_pedestal_widths
  COMMON bm_pedestal_sample_block, bm_info, bm_samples, bm_max_sample_lists, bm_pedestal_centres, bm_channel_centres, bm_pedestal_posns, bm_pedestal_widths

  ; first time check that calibration data has been set up
  ; NB is set up in rcs_initialize_common

  IF NOT KEYWORD_SET(nm_info) THEN rcs_setup_pedestal_calibration, sc

  ; check mode

  IF KEYWORD_SET(bm) THEN BEGIN

    ; check if already calculated

    IF sc EQ bm_info.sc AND bm_info.done[detector,lut] AND NOT KEYWORD_SET(force) THEN BEGIN

      ; get from COMMON block

      samples = bm_samples[*,*,detector,lut]

      max_sample_lists = bm_max_sample_lists[*,*,detector,lut]

      pedestal_posn = bm_pedestal_posns[detector,lut]

      pedestal_width = bm_pedestal_widths[detector,lut]

      ; get pedestal centre

      pedestal_centre = bm_pedestal_centres[detector,lut]

      ; get channel centres

      channel_centres = bm_channel_centres[*,detector,lut]

    ENDIF ELSE BEGIN

      ; check have right calibration info for cross_correlation

      IF bm_info.sc NE sc THEN rcs_setup_pedestal_calibration, sc

      ; calculate sampling

      ies_pedestal_samples, 4, bm_info.nshifts, bm_info.offset, 1+detector, $
        bm_info.lut_map[lut], 0, bm_info.ch_positions[*,*,detector,lut], $
        samples, max_sample_lists, pedestal_posn, pedestal_width, pedestal_centre, channel_centres

      ; set up COMMON block

      bm_samples[*,*,detector,lut] = samples

      bm_max_sample_lists[*,*,detector,lut] = max_sample_lists

      bm_pedestal_posns[detector,lut] = pedestal_posn

      bm_pedestal_widths[detector,lut] = pedestal_width

      bm_pedestal_centres[detector,lut] = pedestal_centre

      bm_channel_centres[*,detector,lut] = channel_centres

      bm_info.done[detector,lut] = 1

    ENDELSE

  ENDIF ELSE BEGIN

    ; check if already calculated

    IF sc EQ nm_info.sc AND nm_info.done[detector,lut]  AND NOT KEYWORD_SET(force) THEN BEGIN

      ; get from COMMON block

      samples = nm_samples[*,*,detector,lut]

      max_sample_lists = nm_max_sample_lists[*,*,detector,lut]

      pedestal_posn = nm_pedestal_posns[detector,lut]

      pedestal_width = nm_pedestal_widths[detector,lut]

      ; get pedestal centre

      pedestal_centre = nm_pedestal_centres[detector,lut]

      ; get channel centres

      channel_centres = nm_channel_centres[*,detector,lut]

    ENDIF ELSE BEGIN

      ; check have right calibration info for cross_correlation

      IF nm_info.sc NE sc THEN rcs_setup_pedestal_calibration, sc

      ; calculate sampling

      ies_pedestal_samples, 2, nm_info.nshifts, nm_info.offset, 1+detector, $
        nm_info.lut_map[lut], 0, nm_info.ch_positions[*,*,detector,lut], $
        samples, max_sample_lists, pedestal_posn, pedestal_width, pedestal_centre, channel_centres

      ; set up COMMON block

      nm_samples[*,*,detector,lut] = samples

      nm_max_sample_lists[*,*,detector,lut] = max_sample_lists

      nm_pedestal_posns[detector,lut] = pedestal_posn

      nm_pedestal_widths[detector,lut] = pedestal_width

      nm_pedestal_centres[detector,lut] = pedestal_centre

      nm_channel_centres[*,detector,lut] = channel_centres

      nm_info.done[detector,lut] = 1

    ENDELSE

  ENDELSE

END