;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PEDESTALS
;
; Purpose     : Locates pedestal bin position and optionally subtracts pedestal from each sample.
;
; Explanation :	Associated with each data sample is a LUT number which defines the
;               bin positions of the y samples. This routine calculates pedestal shift required to a
;               fraction of a channel by cross-correlating the known pedestal profile with the sampled data.
;               It also uses  a centroid algorithm for comparison.
;               The pedestal contribution may then be subtracted if required. It is assumed that the input data
;               is undivided by energy and not summed over spins.
;
;               For RAPID use a normalization factor calculated to give the correct number of pedestal counts
;               given the integration time; the accumulation time and the signal counts observed.
;
; Use         : < pedestals = rcs_pedestals(sc, npeds, array, times, lut_list, nsectors, detector_list, SUB=sub) >
;
; Inputs      : sc            : STRING character giving spacecraft ID
;               npeds         : INT no of pedestal channels -- use to determine if NM or BM
;               array         : FLTARR( nbands, ndetectors, nsectors*npoints) containing input data
;               times         : DBLARR(npoints) TAI times of samples
;               lut_list      : STRUCTURE ARRAY containing LUT run info.
;               nsectors      : INT no. of sectors.
;               detector_list : INTARR[] list of detectors required : 0-8
;
; Opt. Inputs : None.
;
; Outputs     : pedestals : FLTARR ( 5, ndetectors, npoints*nsectors)
;                           0 : pedestal bin position measured by cross_correlation algorithm
;                           1 : pedestal shift from central bin for cross_correlation algorithm
;                           2 : pedestal shift from central bin for centroid algorithm
;                           3 : occupancy defined as the ratio of non-pedestal counts to total counts expressed as a percentage
;                           4 : visible counts defined as ratio of counts in data channels vs expected total counts
;                               expressed out of 10 as missing counts
;
; Opt. Outputs:	None.
;
; Keywords    : SUB          : indicates whether background should be subtracted or not from signal channels
;               BM           : FLAG indicating BM rather than NM.
;
; Written     :	Version 0.0, 17/09/00
;
; Modified    :	Version 0.1, 9/2/01
;                 Converted occupancy and missing counts to percentage and out of 10 respectively.
;               Version 0.2, 20/07/01
;                 Used ies_runs.
;               Version 0.3, 14/05/02
;                 Used rcs_check_ies to get expected counts.
;               Version 0.4, 24/10/03
;                 Changed to lut_list argument.
;               Version 0.5, 20/11/03
;                 Separated calibration info -- removed keywords CH_POSITIONS, COUNT_CAL, ENERGY_CAL, ENERGY_EFF
;                 Used rcs pedestal samples.
;                 Added BM keyword.
;                 Removed ped_chans.
;                 Used ies do cross correlation.
;                 Added sc to argument list and added rcs get pedestal calibration
;               Version 0.6, 04/12/03
;                 Renamed rcs get pedestal calibration.
;                 Added sc argument to rcs pedestal samples.
;                 Moved precalculation out of this routine.
;               Version 0.7, 15/12/03
;                 Fixed bug processing multiple sectors.
;               Version 0.8, 20/01/04
;                 Rearranged processing in ies do cross correlation.
;
; Version     :	Version 0.8, 20/01/04
;-
;**********************************************************

FUNCTION rcs_pedestals, sc, npeds, array, times, lut_list, nsectors, detector_list, SUB=sub, BM=bm

  ; get no of samples

  npoints = N_ELEMENTS(times)

  ; set up no of detectors

  ndetectors = N_ELEMENTS(detector_list)

  ; set up pedestal info output

  pedestals = FLTARR(5, ndetectors, npoints*nsectors)

  FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run

    lut = lut_list[l].lut

    ; get expected total no. of counts
    ; these nos are derived from histogram calibration data

    tot_exp_counts = rcs_check_ies(lut) / FLOAT(nsectors)

    ; loop over each detector

    FOR d = 0, ndetectors-1 DO BEGIN

      ; get calibration info samples for cross_correlation
      ; qsamples[npeds,nshifts] is an array containing shifted pedestal calibration data sampled as for the real data
      ; only the first npeds energy channels are kept
      ; do not search for calibration data at specific temperature
      ; this is used to restrict the search for max cross-correlation
      ; to samples with the same peak position as the data

      rcs_pedestal_samples, sc, detector_list[d], lut, samples, max_sample_lists, pedestal_centre, channel_centres, $
                            pedestal_posn, BM=bm

      ; loop over each sample getting pedestal info

      FOR k = nsectors*lut_list[l].st, nsectors*(lut_list[l].en+1)-1 DO BEGIN

        pp = ies_do_cross_correlation(npeds, array[*,d,k], dummy_ch_positions, max_sample_lists, samples, tot_exp_counts, PED=ped)

        ; subtract pedestals

        IF KEYWORD_SET(sub) THEN array[0:npeds-1,d,k] = (array[0:npeds-1,d,k] - ped) > 0

        pedestals[*,d,k] = rcs_pedestal_info ( ped, pp, array[*,d,k], tot_exp_counts, pedestal_centre, channel_centres, pedestal_posn)

      ENDFOR ; loop over k

    ENDFOR ; loop over d

  ENDFOR ; loop over l

  RETURN, pedestals

END