;+
; Project     :	CLUSTER - PEACE
;
; Name        :	RCS_SETUP_PEDESTAL_CALIBRATION
;
; Purpose     : Retrieves pedestal calibration data and sets up pedestal samples.
;
; Explanation :	Pedestal calibration data is stored in IES_CALIBRATION files.
;               This routine restores the relevant pedestal calibration data provided it has not already
;               been stored and stores in the pedestal calibration data COMMON block.
;               It then calculates for each LUT the counts in the pedestal channels for different positions of the pedestal
;               data.
;
; Use         : < rcs_setup_pedestal_calibration, sc >
;
; Inputs      : sc : STRING character giving S/C
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs:	None.
;
; Keywords    : FORCE : FLAG indicating should force set up of calibration data even if already exists.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/10/00
;
; Modified    :	Version 0.1, 02/10/03
;                 Changed argument list of ies get pedestal calibration
;               Version 0.2, 21/11/03
;                 Removed ch_positions etc from COMMON block.
;                 Used rcs get sampling.
;               Version 0.3, 01/12/03
;                 Changed argument list of rcs get luts.
;                 Renamed from rcs get pedestal calibration.
;                 Renamed rcs get sampling to ies get sampling.
;                 Added pedestal_widths and detectors to argument list of ies get sampling.
;               Version 0.4, 23/01/04
;                 Changed max_sample_lists to INTARR (previously FLTARR)
;               Version 0.4, 02/02/04
;                 Added FORCE keyword.
;                 Added pedestal_width to COMMON block.
;
; Version     :	Version 0.4, 02/02/04
;-
;**********************************************************

PRO rcs_setup_pedestal_calibration, sc, FORCE=force

  COMMON nm_pedestal_sample_block, nm_info, nm_samples, nm_max_sample_lists, nm_pedestal_centres, nm_channel_centres, nm_pedestal_posns, nm_pedestal_widths
  COMMON bm_pedestal_sample_block, bm_info, bm_samples, bm_max_sample_lists, bm_pedestal_centres, bm_channel_centres, bm_pedestal_posns, bm_pedestal_widths

  ; set up COMMON block and restore calibration data

  ; have pedestal calibration data if have nm_info

  ies_get_pedestal_calibration, rcs_get_cal_name(sc), FORCE=force

  ; get bin positions and calibration factors for E3DD data

  nm_ch_positions = rcs_get_luts ( sc, 'E3DD', dummy_lut_times, nm_lut_map, dummy_count_calibration, dummy_energy_calibration, dummy_energy_efficiency, $
                                     NO_OF_DETECTORS=no_of_detectors, NO_OF_LUTS=no_of_luts)

  bm_ch_positions = rcs_get_luts ( sc, 'E3DDBM', dummy_lut_times, bm_lut_map, dummy_count_calibration, dummy_energy_calibration, dummy_energy_efficiency, $
                                     NO_OF_DETECTORS=no_of_detectors, NO_OF_LUTS=no_of_luts)

  ; set up info structures which defines what calculations are stored in COMMON block

  nm_info = {sc:sc, ch_positions:nm_ch_positions, lut_map:nm_lut_map, done:INTARR(no_of_detectors, no_of_luts), nshifts:251, offset:50}
  bm_info = {sc:sc, ch_positions:bm_ch_positions, lut_map:bm_lut_map, done:INTARR(no_of_detectors, no_of_luts), nshifts:251, offset:50}

  ; set up samples array

  nm_samples = FLTARR( 2, nm_info.nshifts, no_of_detectors, no_of_luts)
  bm_samples = FLTARR( 4, bm_info.nshifts, no_of_detectors, no_of_luts)

  ; set up max_sample_lists array
  ; list of shifts for each pedestal channel maxima position
  ; 1st element contains no of samples
  ; these are used as look up tables to speed up search for shifted
  ; samples with the same peak as the actual data

  nm_max_sample_lists = INTARR(1+nm_info.nshifts, 2, no_of_detectors, no_of_luts)
  bm_max_sample_lists = INTARR(1+bm_info.nshifts, 4, no_of_detectors, no_of_luts)

  ; set up pedestal positions

  nm_pedestal_posns = FLTARR( no_of_detectors, no_of_luts)
  bm_pedestal_posns = FLTARR( no_of_detectors, no_of_luts)

  ; set up pedestal widths

  nm_pedestal_widths = FLTARR( no_of_detectors, no_of_luts)
  bm_pedestal_widths = FLTARR( no_of_detectors, no_of_luts)

  ; set up pedestal centres

  nm_pedestal_centres = FLTARR( no_of_detectors, no_of_luts)
  bm_pedestal_centres = FLTARR( no_of_detectors, no_of_luts)

  ; set up channel centres

  nm_channel_centres = FLTARR( 2, no_of_detectors, no_of_luts)
  bm_channel_centres = FLTARR( 4, no_of_detectors, no_of_luts)

END