;+
; Project     :	CLUSTER - PEACE
;
; Name        :	RCS_TIME_AVERAGE
;
; Purpose     :	Produces time resampled data
;
; Explanation :	Resamples over fixed time spans.
;               If no data in a time span then value is zero.
;               POINTS gives no of points used in each resample.
;               Returns 1 if OK else 0.
;               Default PERIOD is one hour.
;
; Use         : < res = rcs_time_average( time_start, time_end, time, data, atime, adata, $
;                   PERIOD=period, POINTS=points, MEDIAN=median, THRESHOLD=threshold)
;
; Inputs      : time_start : DOUBLE TAI start time of ROI
;               time_end   : DOUBLE TAI stop time  of ROI
;               time       : DBLARR[nsamples] TAI start times of sample.
;               data       : FLTARR[ndetectors,nbands,nsamples] data
;
; Opt. Inputs : None.
;
; Outputs     : Returns 1 if OK else 0.
;
; Opt. Outputs: atime : DBLARR[] TAI time of each resampled data point (start time).
;               adata : FLTARR[] resampled data.
;
; Keywords    : PERIOD : LONG time span in seconds. Only truncates to one day.
;               POINTS : LONARR[] returning no. of points in each new sample.
;               MEDIAN : FLAG indicating should apply median filter rather than average.
;               THRESHOLD : FLAG indicating should check if any dat aabove threshold before averaging
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 5/3/01
;                 Changed averaging procedure.
;               Version 0.2, 13/06/01
;                 Changed averaging procedure.
;                 Dealt with anp=0
;               Version 0.3, 18/01/02
;                 Corrected INT16 bug.
;               Version 0.4, 21/03/02
;                 Added MEDIAN keyword.
;               Version 0.5, 12/04/02
;                 Changed so that returns array for adata if no data.
;               Version 0.6, 03/05/02
;                 Added THRESHOLD keyword.
;               Version 0.7, 04/01/03
;                 Updated documentation.
;                 Changed routine to FUNCTION.
;
; Version     : Version 0.7, 04/01/03
;-
;**********************************************************

FUNCTION rcs_time_average, time_start, time_end, time, data, atime, adata, $
    PERIOD=period, POINTS=points, MEDIAN=median, THRESHOLD=threshold

  IF NOT KEYWORD_SET(period) THEN period = 3600L

  ; assume period is less than or equal to one day
  ; to avoid problems with leap seconds use MJD format

  tmjd = tai2utc(time_start)

  period_ms = period*1000L

  tmjd.time = (tmjd.time/period_ms)*period_ms

  tstart = utc2tai(tmjd)

  ; set up new samples

  anp = LONG((time_end-tstart)/period+0.99999)

  IF anp EQ 0 THEN RETURN, 0

  atime = tstart+DINDGEN(anp)*period

  IF NOT KEYWORD_SET(data) THEN BEGIN

    adata = LONARR(anp)

    RETURN, 0

  ENDIF

  ndetectors = (SIZE(data))[1]
  nbands = (SIZE(data))[2]

  adata = FLTARR( ndetectors, nbands, anp )

  points = LONARR(anp)

  ; get new samples

  k = 0L

  kstart = 0

  FOR n = 0L, anp-1 DO BEGIN

    WHILE (time[k] LT atime[n]+period) AND (k LT N_ELEMENTS(time)-1) DO k = k + 1

    kpoints = k - kstart

    IF kpoints EQ 1 THEN BEGIN

      adata[*,*,n] = data[*,*,kstart]

    ENDIF ELSE IF kpoints GT 1 THEN BEGIN

      IF KEYWORD_SET(threshold) THEN BEGIN

        ; checks if any data above threshold before averaging

        list = WHERE(data[*,*,kstart:k-1] GT threshold, count)

        IF count EQ 0 THEN adata[*,*,n] = TOTAL(data[*,*,kstart:k-1],3)/kpoints

      ENDIF ELSE IF KEYWORD_SET(median) THEN BEGIN

        FOR d = 0, ndetectors-1 DO adata[d,*,n] = MEDIAN(data[d,*,kstart:k-1],/EVEN)

      ENDIF ELSE BEGIN

        adata[*,*,n] = TOTAL(data[*,*,kstart:k-1],3)/kpoints

      ENDELSE

    ENDIF

    points[n] = kpoints

    kstart = k

  ENDFOR

  RETURN, 1

END
