;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_ADD_DATA
;
; Purpose     :	Sets up RAL Intermediate format data structure.
;
; Explanation : Creates RAL Intermediate format data structure and slots new
;               data sample into current array ie does not assume time ordered.
;
; Use         : < rcs_add_data, experiment, sc, vs, data_type, title, no_of_sectors, time, header, data, newdata, $
;                               lut_no, flag, sample_time, MVALS=mvals, PTITLE=ptitle, ZTITLE=ztitle >
;
; Inputs      : experiment    : STRING goes into header and data structure names and equals header.datatype.
;               sc            : STRING character giving spacecraft ID
;               vs            : STRING IF version 'sss_v_a_b'
;               data_type     : STRING giving IF data set as used by rcs get luts.
;               title         : STRING giving header.title
;               no_of_sectors : INT no of sectors in IF data set
;               time          : DOUBLE TAI time of start of sample
;               header        : STRUCTURE IF data set header
;               data          : STRUCTURE ARRAY IF data set data
;               newdata       : FLTARR ( nsectors, ndetectors, nenergies )  data for next sample (NB may be zero to initialize next sample)
;               lut_no        : INT sample LUT number : 0-3.
;               flag          : LONG sample flag
;               sample_time   : DOUBLE nominal time in seconds spanned by sample.
;                               If start time of next sample beyond this then sample_time will define sample duration.
;
; Opt. Inputs : None.
;
; Outputs     : n : LONG no. of elements of Intermediate format array.
;
; Opt. Outputs: None.
;
; Keywords    : MVALS   : BYTARR M values.
;               PTITLE  : STRARR header.ptitle
;               ZTITLE  : STRING header.ztitle
;               EXDATA  : STRING extra datatype information
;               OFFSET  : LONG position in data.data array
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/09/00
;
; Modified    :	Version 0.1, 3/2/01
;                 Added no_of_luts to EXPERIMENT which is used in structure name.
;                 This prevents changing no of luts causing conflicting structures.
;               Version 0.2, 09/05/01
;                 Added error flag if endtime adjusted by more than 1 second.
;               Version 0.3, 25/05/01
;                 Added EXDATA keyword.
;               Version 0.4, 18/07/01
;                 Added version to header structure.
;               Version 0.5, 18/07/01
;                 Added CDROM version number.
;               Version 0.6, 08/07/02
;                 Added KEYWORD OFFSET.
;                 Used separate routine rcs_increase_storage
;               Version 0.7, 19/02/03
;                 Removed some diagnostic checking.
;               Version 0.8, 03/09/03
;                 Changed vs input to format sss_v_a_b (previously sv)
;               Version 0.9, 28/11/03
;                 Added '_' to exdata.
;               Version 1.0, 01/12/03
;                 Chanegd argument list of rcs get luts.
;
; Version     :	Version 1.0, 01/12/03
;-
;**********************************************************

PRO rcs_add_data, experiment, sc, vs, data_type, title, no_of_sectors, time, header, data, newdata, lut_no, flag, sample_time, $
                  MVALS=mvals, PTITLE=ptitle, ZTITLE=ztitle, EXDATA=exdata, OFFSET=offset

  ; set up internal flag

  tflag = flag

  ; set up header if does not exist

  IF NOT KEYWORD_SET(header) THEN BEGIN

    ; set up ztitle

    IF NOT KEYWORD_SET(ztitle) THEN ztitle = 'Counts'

    ; set exdata

    IF NOT KEYWORD_SET(exdata) THEN xdata = '' ELSE xdata = '_' + exdata

    ; get bin positions and calibration factors

    ch_positions = rcs_get_luts ( sc, data_type, lut_times, lut_map, count_calibration, energy_calibration, energy_efficiency, $
                   NO_OF_ENERGIES=no_of_energies, NO_OF_DETECTORS=no_of_detectors, NO_OF_LUTS=no_of_luts)

    ; set up output structure
    ; NB the brackets are necessary to preserve array character if ndetectors=1

    header = ies_header_structure ( experiment, 0, no_of_detectors, no_of_energies, no_of_sectors, no_of_luts, $
                                     EXPERIMENT = experiment+xdata+'_'+rcs_strtrim(no_of_luts), $
                                     VERSION    = vs, $
                                     TIME_START = time, $
                                     TIME_STOP  = time, $
                                     TITLE      = title,  $
                                     XTITLE     = 'Time', $
                                     YTITLE     = 'Energy', $
                                     ZTITLE     = ztitle, $
                                     ENERGY_EFFICIENCY=energy_efficiency, $
                                     ENERGY_CALIBRATION=energy_calibration, $
                                     COUNT_CALIBRATION=count_calibration, $
                                     LUT_MAP = lut_map, LUT_TIMES=lut_times, PTITLE=ptitle )

    header.ch_positions = ch_positions

    IF data_type EQ 'EPAD2' THEN header.ndetectors = 3

    ; set up data structure

    data   = ies_data_structure ( 1, header.ndetectors, header.nbands, header.nsectors, EXPERIMENT=experiment+xdata, MVALS=mvals)

  ENDIF

  ; increase size of data array if necessary and possible
  ; can get into a loop here because needs temporary array
  ; prevent by using in place algorithm

  IF header.npoints EQ N_ELEMENTS(data) THEN rcs_increase_storage, header, data

  ; set up index and times for new datum
  ; slot datum into data array
  ; do not assume that goes at end
  ; add end time to previous data point
  ; adjust if time greater than 8 seconds

  IF header.npoints EQ 0 THEN BEGIN

    ; first sample
    ; set index into data for new sample

    index = 0

    ; set times

    data[index].time = time

    data[index].endtime = time + sample_time

    header.tstop = data[index].endtime

  ENDIF ELSE IF time GE data[header.npoints-1].time THEN BEGIN

    ; simply add to end
    ; set index into data for new sample

    index = header.npoints

    ; set times

    data[index-1].endtime = data[index-1].endtime < time

    data[index].time = time

    data[index].endtime = time + sample_time

    header.tstop = data[index].endtime

  ENDIF ELSE IF time LT data[0].time THEN BEGIN

    ; add to beginning
    ; shift data array

    data[1:header.npoints] = data[0:header.npoints-1]

    ; set index into data for new sample

    index = 0

    ; check if sample time needs adjusting

    adjustment = (time + sample_time - data[1].time) > 0.0d0

    ; set flag if adjustment too large

    IF adjustment GT 1.0D0 THEN tflag = tflag AND '40'X

    ; set times

    data[index].time = time

    data[index].endtime = time + sample_time - adjustment

    header.tstart = time

  ENDIF ELSE BEGIN

    ; slot in the middle
    ; shift data array

    list = WHERE(data.time GT time)

    data[list[0]+1:header.npoints] = data[list[0]:header.npoints-1]

    ; set index into data for new sample

    index = list[0]

    ; set times

    ; check if previous sample time needs adjusting

    adjustment = (data[index-1].endtime - time) > 0.0d0

    ; set flag if adjustment too large

    IF adjustment GT 1.0D0 THEN data[index-1].flag = data[index-1].flag AND '40'X

    data[index-1].endtime = data[index-1].endtime - adjustment

    ; check if current sample time needs adjusting

    adjustment = (time + sample_time - data[index+1].time) > 0.0d0

    ; set flag if adjustment too large

    IF adjustment GT 1.0D0 THEN tflag = tflag AND '40'X

    data[index].endtime = time + sample_time - adjustment

  ENDELSE

  ; get lut adjusted for changes
  ; unchanged for IIMS or EHIST
  ; NB do not propagate change up argument list

  lut = rcs_adjust_lut (time, lut_no, header.lut_times)

  ; extract e-3dd nm data and decompress
  ; 8 energies, 9 detectors, spin averaged
  ; data:FLTARR(no_of_sectors,no_of_detectors,no_of_energies)

  IF KEYWORD_SET(offset) THEN $
    data[index].data[offset:*] = newdata $
  ELSE $
    data[index].data = newdata

  ; set lut

  data[index].lut = lut

  ; set flag

  data[index].flag = tflag

  ; set m and msigns

  IF KEYWORD_SET(mvals) THEN data[index].mvals = mvals

  ; increment no. of points

  header.npoints = header.npoints + 1

  RETURN

END
