;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_ADD_I3DD
;
; Purpose     : Adds IIMS I3DD data to I3DD IF data set.
;
; Explanation : Adds IIMS I3DD data to I3DD IF data set.
;
; Use         : < rcs_add_i3dd, sc, vs, time, i3dd, subc, flag  >
;
; Inputs      : sc          : STRING character giving spacecraft ID
;               vs          : STRING IF version 'sss_v_a_b'
;               time        : DOUBLE TAI time of start of EDB
;               i3dd        : FLTARR ( nsectors, ndetectors, nenergies )  i3dd counts
;               subc        : INT subcommutation index
;               flag        : BYTE flag bits
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, 08/05/01
;                 Corrected accumulation time for sc2.
;               Version 0.2, 18/07/01
;                 Added CDROM version number.
;               Version 0.3, 29/09/03
;                 Added SC to title.
;
; Version     :	Version 0.3, 29/09/03
;-
;**********************************************************

PRO rcs_add_i3dd, sc, vs, time, i3dd, subc, flag

  ; global common block

  COMMON rcs_i3dd_H_block, header_H, data_H

  COMMON rcs_i3dd_He_block, header_He, data_He

  COMMON rcs_i3dd_CNO_block, header_CNO, data_CNO

  ; check SC to determine accumulation time in secs

  IF sc EQ '2' THEN acc = 4.0 ELSE acc = 32.0

  no = 32

  subc = subc MOD no

  tflag = flag

  new_sample = rcs_subc( header_H, data_H, subc, no, tflag)

  tflag = flag

  new_sample = rcs_subc( header_He, data_He, subc, no, tflag)

  tflag = flag

  new_sample = rcs_subc( header_CNO, data_CNO, subc, no, tflag)

  ; create new samples with zeroes
  ; if new set of data or missing data packets

  IF new_sample THEN BEGIN

    ; create new data sample
    ; cancel bad LUT flag since irrelevant to IIMS

    rcs_add_data, rcs_get_scname(sc)+'_IIMS_I3DD_H', sc, vs, 'I3DD_H', rcs_get_scname(sc)+' I-3DD H', 16, time, header_H, data_H, 0.0, 0, tflag AND 'EF'X, acc

    rcs_add_data, rcs_get_scname(sc)+'_IIMS_I3DD_He', sc, vs, 'I3DD_He', rcs_get_scname(sc)+' I-3DD He', 16, time, header_He, data_He, 0.0, 0, tflag AND 'EF'X, acc

    rcs_add_data, rcs_get_scname(sc)+'_IIMS_I3DD_CNO', sc, vs, 'I3DD_CNO', rcs_get_scname(sc)+' I-3DD CNO', 16, time, header_CNO, data_CNO, 0.0, 0, tflag AND 'EF'X, acc

  ENDIF

  ; insert data[ndetectors,nenergies,m,nsectors] into data.data[no_of_sectors,no_of_detectors,no_of_energies,no_of_points]

  sector = subc/2

  ; check which masses

  IF (subc MOD 2) EQ 0 THEN BEGIN

    ; H and He/2

    data_H[header_H.npoints-1].data[sector,*,*] = i3dd[0:95]

    data_He[header_He.npoints-1].data[sector,*,0:3] = i3dd[96:143]

  ENDIF ELSE BEGIN

    ; He/2 and CNO

    data_He[header_He.npoints-1].data[sector,*,4:7] = i3dd[0:47]

    data_CNO[header_CNO.npoints-1].data[sector,*,*] = i3dd[48:143]

  ENDELSE

END
