;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	rcs_calculate_luts
;
; Purpose     :	Get start channel positions for each detector for the different lookup tables.
;
; Explanation : Get start channel positions for each detector for the different lookup tables and times when LUTs changed.
;
; Use         : < luts = rcs_calculate_luts( sc, lut_times) >
;
; Inputs      : experiment      : String giving experiment
;               .
; Opt. Inputs : None.
;
; Outputs     : luts      : BYTARR (no_of_energies+1, no_of_detectors, no_of_luts)
;                           bottom bin positions of energy channels for each detector,
;               lut_times : DBLARR(no_of_changes) times of LUT changes
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/09/00
;
; Modified    :	Version 0.1, 1/2/01
;                 Corrected code for multiple lut times.
;               Version 0.2, 15/11/02
;                 Changed so that returns only start boundary of bin instead of start and end.
;                 Previously returned array [2,no_of_energies,no_of_detectors,no_of_luts].
;                 Now returns array [no_of_energies+1,no_of_detectors,no_of_luts].
;                 NB have extra bin at end set at 256 to simplify subsequent processing.
;
; Version     :	Version 0.2, 15/11/02
;-
;**********************************************************

FUNCTION rcs_calculate_luts, sc, lut_times

  ; luts = FLTARR(nbands,ndetectors,nluts)
  ; where nluts = 4*nchanges
  ; and lutimes= DBLARR(nchanges)

  CASE sc OF
    '1' : lut_times = rcs_get_f8 (p, s, b)
    '2' : lut_times = rcs_get_f1 (p, s, b)
    '3' : lut_times = rcs_get_f6 (p, s, b)
    '4' : lut_times = rcs_get_f7 (p, s, b)
    '9' : lut_times = rcs_get_f9 (p, s, b)
    ELSE : lut_times = rcs_get_ph (p, s, b)
  ENDCASE

  nchanges = N_ELEMENTS(lut_times)

  luts = FLTARR(17,9,4*nchanges)

  ; set up start boundaries
  ; NB have p and s INTARR(9,4,n) or INTARR(9,4) if n=1

  luts[0,*,*] = 0
  luts[1,*,*] = 1
  luts[2,*,*] = p-2*s
  luts[3,*,*] = p-s
  luts[4,*,*] = p
  luts[5,*,*] = p+s
  luts[6,*,*] = p+2*s
  FOR n=0,nchanges-1 DO FOR k = 7, 14 DO luts[k,*,4*n:4*n+3] = p[*,*,n]+b[k-7,n]
  luts[15,*,*] = 255
  luts[16,*,*] = 256

  ; ensure bins do not underflow

  FOR k = 1, 15 DO luts[k,*,*] = luts[k,*,*] > (luts[k-1,*,*]+1)

  ; ensure bins do not overflow

  luts[15,*,*] = 255

  FOR k = 1, 15 DO luts[15-k,*,*] = luts[15-k,*,*] < (luts[16-k,*,*]-1)

  RETURN, luts

END
