;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_CHECK_EDB
;
; Purpose     :	Checks EDB packet and extracts some flags.
;
; Explanation : Checks EDB packet and extracts some flags.
;
; Use         : < res = rcs_check_edb(edb, time, type, hist_mode, flag, $
;                  lut_no, epad_table, fgm, SUBC_INVALID=subc_invalid) >
;
; Inputs      : edb     : BYTARR EDB packet
;               time    : DOUBLE TAI time of EDB
;               type    : INT indicates telemetry mode of packet
;                           0 : nm
;                           1 : bm1
;                           2 : bm2
;                           3 : bm3
;
; Opt. Inputs : None.
;
; Outputs     : res  : INT result
;                        0 : OK to process packet
;                        1 : IFFT
;                        2 : EPP TEST
;                        3 : RAM CHECK
;                        4 : CLASS TEST
;                        5 : IES-IF INACTIVE
;                        6 : EPP TEST ACTIVE
;               hist_mode  : FLAG indicating IES histogram mode active
;               flag       : BYTE flag bits
;                              set flag use bits 1 and 2 for compatability with intermediate plot flags
;                              bit 1 : 0 = NM ; 1 = BM
;                              bit 2 : 0 = normal ; 1 = abnormal
;                              bit 4 : lut invalid
;                              bit 5 : subc invalid
;
;               lut_no     : INT IES LUT 0:3 (0:8 in histogram mode)
;                             -1 indicates invalid LUT
;               epad_table : FLAG indicating epad table A (0) or B (1)
;               fgm        : FLAG indicating whether mag field available
;                            in all sectors
;
; Opt. Outputs: None.
;
; Keywords    : SUBC_INVALID : FLAG indicating subcommutation index jump
;               GAP_ERROR    : FLAG indicating next packet does not follow consecutively.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, 13/12/00
;                 Added SILENT check.
;               Version 0.2, 12/2/01
;                  Removed silent keyword into rcs_text,
;               Version 0.3, 14/03/01
;                 Removed ids from rcs_text and this argument list.
;               Version 0.4, 04/04/01
;                 Updated description.
;                 Added keyword GAP_ERROR.
;               Version 0.5, 22/04/01
;                 Changed name of COMMON block.
;                 Added erdifind qualifier.
;               Version 0.6, 12/02/03
;                 Used rcs_check_bm.
;               Version 0.7, 12/02/03
;                 Added check that histogram gives valid detector else gives warning.
;               Version 0.8, 07/10/03
;                 Corrected bug which converted epad_table to INT rather than BYTE
;               Version 0.9, 08/11/03
;                 Removed duplicate check on lut_no.
;                 Removed debugging printout.
;
; Version     :	Version 0.9, 08/11/03
;-
;**********************************************************

FUNCTION rcs_check_edb, edb, time, type, hist_mode, flag, lut_no, epad_table, fgm, $
  SUBC_INVALID=subc_invalid, GAP_ERROR=gap_error

  ; global common block

  COMMON rcs_hk_qualifier_block, current_lut_no, current_erdifind

  ; local COMMON block

  COMMON local_hist_block, short_hist_mode, long_hist_mode

  ; set flag use bits 1 and 2 for compatability with intermediate plot flags
  ; bit 1
  ;     0 = NM
  ;     1 = BM
  ; bit 2
  ;     0 = normal
  ;     1 = abnormal

  flag = ISHFT(type,1)

  IF KEYWORD_SET(subc_invalid) THEN flag = flag OR '20'X

  IF KEYWORD_SET(gap_error) THEN flag = flag OR '40'X ; put error in bad times for now

  ; set invalid lut_no and LUTX flag

  lut_no = -1

  flag = flag OR '10'X

  ; check type

  IF rcs_check_bm(type) THEN BEGIN

    ; BM1/BM3

    ; get content descriptors

    cd1 = edb['04'X]
    cd2 = edb['154'X]

    ; check 2nd sync marker
    ; NB not in DARD but 2nd sync = 4c e2 ff

    IF edb['212'X] NE '4C'X OR edb['213'X] NE 'E2'X OR edb['214'X] NE 'FF'X THEN BEGIN

      rcs_text, 'BM 2ND SYNC FAILED' + $
                STRING(edb['212'X:'214'X],FORMAT='(3Z3)')

    ENDIF

    ; check 3rd sync marker

    IF edb['475'X] NE '54'X OR edb['476'X] NE 'E1'X OR edb['477'X] NE 'E1'X THEN BEGIN

      rcs_text, 'BM 3RD SYNC FAILED' + $
                STRING(edb['475'X:'477'X],FORMAT='(3Z3)')

    ENDIF

  ENDIF ELSE BEGIN

    ; NM/BM2

    ; get content descriptors

    cd1 = edb['04'X]
    cd2 = edb['14F'X]

  ENDELSE

  ; check content descriptors

  ; EPP test
  ; RAM check

  IF (cd1 AND 'F'X) NE 0 THEN BEGIN

    IF (cd1 AND '1'X) NE 0 THEN BEGIN
      rcs_text, 'IFFT'
      RETURN, 1
    ENDIF

    IF (cd1 AND '2'X) NE 0 THEN BEGIN
      rcs_text, 'EPP TEST'
      RETURN, 2
    ENDIF

    IF (cd1 AND '4'X) NE 0 THEN BEGIN
      rcs_text, 'RAM CHECK'
      RETURN, 3
    ENDIF

    IF (cd1 AND '8'X) NE 0 THEN BEGIN
      rcs_text, 'CLASS TEST'
      RETURN, 4
    ENDIF

  ENDIF

  ; IES-IF ACTIVE

  IF (cd1 AND '40'X) EQ 0 THEN BEGIN

    rcs_text, 'IES-IF INACTIVE'

    RETURN, 5

  ENDIF

  ; EPP TEST ACTIVE

  IF cd2 AND '40'X THEN BEGIN

    rcs_text, 'EPP TEST ACTIVE'

    RETURN, 6

  ENDIF

  ; check if magnetic field data available in all sectors

  fgm = (cd1 AND '10'X) NE 0

  ; get lut no

  tlut_no = cd2 AND '1F'X

  ; get epad table a = 0 b = 1

  epad_table = ISHFT(cd2 AND '20'XB,1)

  ; check for histogram mode

  IF (cd2 AND '80'X) NE 0 THEN BEGIN

     hist_mode = 1

     subc = edb['003'X]

     odd = subc MOD 2

     even = 1 - odd

     ; get strip no (depends on BM/NM)

     IF rcs_check_bm(type) THEN strip = edb['478'X] ELSE strip = edb['150'X]

     strip = strip AND '0F'XB ; NB not in DARD but see C.Dierker

     ; check lut_no
     ; get some invalid hist mode packets at start of accumulations
     ; in short histogram mode only get one valid spin of data
     ; in hist mode lut no should give strip no
     ; NB data.lut overwritten by HK lut no in histogram mode
     ; NB lut_no = -1 unless reaches end successfully

     ; fix lut_no in short hist mode
     ; in short hist mode :
     ; even packets usually have lut_no-strip = 16
     ; odd  packets usually have lut_no-strip = 17
     ; However, now have a case where lut_no-strip = 17 for both even and odd packet
     ; Strip goes from 1-9, lut_no = 17-26
     ; in long hist mode :
     ; get packets with lut_no-strip=16 but these contain invalid data
     ; strip goes from 0-8, lut_no = 17-25
     ; Need to recognize short_hist_mode and use strip-1 to recognize detector.

     ; reject if invalid lut_no or strip
     ; NB valid ranges are lut_no = 17:26, strip = 0:9

     IF tlut_no LT 17 OR tlut_no GT 26 OR strip GT 9 THEN BEGIN
       short_hist_mode = 0
       RETURN, 0
     ENDIF

     ; reject if previous even packet is not in sequence
     ; Nb in BM only use even packet

     IF KEYWORD_SET(subc_invalid) AND odd THEN RETURN, 0

     ; reject if lut_no and strip are not in agreement

     IF tlut_no - strip LT 16 OR tlut_no-strip GT 17 THEN BEGIN
       short_hist_mode = 0
       RETURN, 0
     ENDIF

     ; 17, 0 is valid in long histogram mode but not in short histogram mode
     ; in short histogram mode get sequence like 16,0 17,0 or 0,0 17,0 followed by
     ; combinations like 17,1 18,1
     ; use sequence 17.0 17,1 18,1 odd even odd to confirm short hist mode
     ; if dont get this then presume long hist mode
     ; this invalidates a valid sample in long hist mode

     IF KEYWORD_SET(long_hist_mode) THEN BEGIN

       ; avoid test for short hist mode

     ENDIF ELSE IF KEYWORD_SET(short_hist_mode) THEN BEGIN

       ; check if actually long hist mode

       CASE short_hist_mode OF
         18 : IF tlut_no NE 17 OR strip NE 1 THEN BEGIN
                short_hist_mode = 0
                long_hist_mode = 1
              ENDIF
         17 : IF tlut_no NE 18 OR strip NE 1 THEN BEGIN
                short_hist_mode = 0
                long_hist_mode = 1
              ENDIF
         ELSE :
       ENDCASE

     ENDIF ELSE BEGIN

       ; check if entering short_hist_mode if 17,0 and odd

       IF tlut_no EQ 17 AND strip EQ 0 AND odd THEN BEGIN

         short_hist_mode = 18

         RETURN, 0

       ENDIF

     ENDELSE

     ; fix lut_no = 17-25 in short hist mode

     IF KEYWORD_SET(short_hist_mode) THEN BEGIN
       tlut_no = strip + 16
       short_hist_mode = short_hist_mode - 1
     ENDIF

     ; check if bad lut_no

     IF tlut_no LT 17 OR tlut_no GT 25 THEN BEGIN
       short_hist_mode = 0
       RETURN, 0
     ENDIF

     ; subtract offset so that lut_no = strip and reset LUTX flag bit

     lut_no = tlut_no-17

     flag = flag AND 'EF'X

     RETURN, 0

  ENDIF ELSE BEGIN

    hist_mode = 0

    long_hist_mode = 0

    ; check lut_no

    IF  (tlut_no GT 3) THEN BEGIN

      rcs_text, 'INVALID LUT : '+rcs_strtrim(tlut_no) + $
                ' type : '+rcs_strtrim(FIX(type)) + $
                ' ' + anytim2cal(time)
      RETURN, 0

    ENDIF ELSE BEGIN

      ; NB ensure current lut no valid and reset flag bit

      current_lut_no = tlut_no

      lut_no = tlut_no

      flag = flag AND 'EF'X

      RETURN, 0

    ENDELSE

  ENDELSE

END
