;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_DECODE_M
;
; Purpose     :	Decodes the mvals array to give detectors used for each sector.
;               The m-value is the polar direction of the perpendicular to the B-field
;               expressed as 0-15 with 0 along the SC spin_axis.
;               The IIMS 3rd direction reflects the B-field perp. direction.
;               The IIMS 1st direction is the direction closest to the B-field.
;               The sense of this is determined from the M-sign.
;               For IES the 1st direction gives direction of the perpendicular B-field.
;               The 2nd direction is close to the direction closest to the B-field.
;               For some reason it is not always exactly the direction closest to the
;               B-field.
;               To determine whether this is parallel or anti-parallel to the B-field
;               need to look at the M-value and the M-sign.
;
; Use         : < tab = RCS_DECODE_M(m) >
;
; Inputs      : m : BYTARR containing the [8]BYTE m values determining the detectors
;                   for each sector, [2]BYTE msigns determining B-field parallel or
;                   anti-parallel to 1st detector in IIMS and [1]BYTE flag indicating EPAD
;                   table A or B.
;
; Opt. Inputs : None.
;
; Outputs     : tab : [3,16]INTARR indicating detector number for each sector.
;
; Opt. Outputs: None.
;
; Keywords    : IIMS : FLAG indicating IIMS else IES
;               MVAL : FLAG indicating should return [16] mvals rather than tab.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 8/1/00
;
; Modified    :	Version 0.1, 12/02/02
;                 Corrected bug in working out m-values associated with each sector.
;               Version 0.2, 06/08/02
;                 Separated into new routine.
;               Version 0.3, 07/10/03
;                 Fixed bug with getting 2nd nibble.
;                 .mvals tag were previously converted from BYTEs to INTs
;
; Version     :	Version 0.2, 06/08/02
;-
;**********************************************************

FUNCTION rcs_decode_m, m, IIMS=iims, MVAL=mval

  ; extract high and low nibbles from m-values
  ; high nibble first followed by low nibble for each byte
  ; NB this previously converted BYTE to INT
  ; Note below creates an array [8,2] then transposes it and reforms it
  ; must be very careful that array [8,2] is created it is easy to end up with a [16] array instead

  mvals = REFORM(TRANSPOSE([[ISHFT(m[0:7],-4)AND 'F'XB],[m[0:7] AND 'F'XB]]),16)

  ; check if MVAL set

  IF KEYWORD_SET(mval) THEN BEGIN

    ; get the 3 detectors for each of the 16 sectors

    RETURN, FIX(mvals)

  ENDIF

  RETURN, rcs_decode_mvals( mvals, IIMS=iims, IES=m[10])

END
