;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_DECODE_MSIGNS
;
; Purpose     :	Decodes the mvals array to give B field orientation for each sector.
;
; Explanation : The m-value is the polar direction of the perpendicular to the B-field
;               expressed as 0-15 with 0 along the SC spin_axis.
;               The IIMS 3rd direction reflects the B-field perp. direction. 
;               The IIMS 1st direction is the direction closest to the B-field.
;               The sense of this is determined from the M-sign.
;               For IES the 1st direction gives direction of the perpendicular B-field.
;               The 2nd direction is close to the direction closest to the B-field.
;               For some reason it is not always exactly the direction closest to the
;               B-field.
;               To determine whether this is parallel or anti-parallel to the B-field
;               need to look at the M-value and the M-sign. 
;
; Use         : < tab = RCS_DECODE_MSIGNS(m) >
;
; Inputs      : m : BYTARR containing the [8]BYTE m values determining the detectors
;                   for each sector, [2]BYTE msigns determining B-field parallel or 
;                   anti-parallel to 1st detector in IIMS and [1]BYTE flag indicating EPAD 
;                   table A or B.
;
; Opt. Inputs : None.
;
; Outputs     : tab : [16]INTARR indicating orientation for each sector.
;                     0 : B-field parallel to the 1st IIMS direction for this sector
;                     1 : B-field is anti-parallel to the 1st IIMS direction 
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 8/1/00
;
; Modified    :	None.
;
; Version     :	Version 0.0, 8/1/00
;-
;**********************************************************

FUNCTION rcs_decode_msigns, mvals

  ; get msigns

  msigns = rcs_fix(mvals[8:9])

  tab = INTARR(16)

  FOR k = 0, 15 DO BEGIN
   
    tab[k] = msigns AND 1

    msigns = ISHFT(msigns,-1)

  ENDFOR

  ; NB ordered as byte0 = sectors 7-0, byte1 = sectors 15-8

  RETURN, [REVERSE(tab[0:7]),REVERSE(tab[8:15])]

END
