;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_DECODE_MVALS
;
; Purpose     :	Decodes the mvals array to give detectors used for each sector.
;               The m-value is the polar direction of the perpendicular to the B-field
;               expressed as 0-15 with 0 along the SC spin_axis.
;               The IIMS 3rd direction reflects the B-field perp. direction. 
;               The IIMS 1st direction is the direction closest to the B-field.
;               The sense of this is determined from the M-sign.
;               For IES the 1st direction gives direction of the perpendicular B-field.
;               The 2nd direction is close to the direction closest to the B-field.
;               For some reason it is not always exactly the direction closest to the
;               B-field.
;               To determine whether this is parallel or anti-parallel to the B-field
;               need to look at the M-value and the M-sign. 
;
; Use         : < tab = RCS_DECODE_MVALS(mvals) >
;
; Inputs      : mvals : [16] m values determining the detectors for each sector.
;
; Opt. Inputs : None.
;
; Outputs     : tab : [3,16]INTARR indicating detector number for each sector.
;
; Opt. Outputs: None.
;
; Keywords    : IIMS : FLAG indicating IIMS else IES
;               IES  : IES table 0 or 1
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 8/1/00
;
; Modified    :	Version 0.1, 12/02/02
;                 Corrected bug in working out m-values associated with each sector.
;               Version 0.2, 06/08/02
;                 Added COMMON block to speed up.
;                 Separated into new routine.
;
; Version     :	Version 0.2, 06/08/02
;-
;**********************************************************

FUNCTION rcs_decode_mvals, mvals, IIMS=iims, IES=ies

  COMMON rcs_decode_m_local, ies_table0, ies_table1, iims_table

  ; check if IIMS

  IF KEYWORD_SET(iims) THEN BEGIN

    IF NOT KEYWORD_SET(iims_table) THEN BEGIN

      ; set up IIMS table [16 mvalues,3]

      iims_table = [ [6,7,7,8,9,10,10,11,0,1,1,2,3,4,4,5], [11,11,11,11,0,0,0,11,11,11,11,11,11,0,0,0],$
                     [0,1,1,2,3,4,4,5,6,7,7,8,9,10,10,11] ]

      ; get in form [3,16]

      iims_table = TRANSPOSE(iims_table)

    ENDIF

    ; get the 3 detectors for each of the 16 sectors

    RETURN, iims_table[*,mvals]

  ENDIF ELSE BEGIN

    IF NOT KEYWORD_SET(ies_table0) THEN BEGIN

      ; set up IES tables [16 mvalues,4]

      ies_table0 = [ [1,1,2,2,3,4,4,5,5,6,6,7,7,8,9,9], [6,6,7,7,8,9,9,1,1,1,1,2,2,3,4,4],$
                     [9,9,9,9,1,1,1,9,9,9,9,9,9,1,1,1] ] - 1

      ies_table1 = [ [1,1,2,2,3,4,4,5,5,6,6,7,7,8,9,9], [6,6,7,7,8,9,9,1,1,1,1,2,2,3,4,4],$
                     [4,4,5,5,6,7,7,9,9,3,3,4,4,5,6,6] ] - 1

      ; get in form [3,16]

      ies_table0 = TRANSPOSE(ies_table0)

      ies_table1 = TRANSPOSE(ies_table1)

    ENDIF

    ; get the 3 detectors for each of the 16 sectors

    IF KEYWORD_SET(ies) THEN BEGIN

      RETURN, ies_table1[*,mvals]

    ENDIF ELSE BEGIN

      RETURN, ies_table0[*,mvals]

    ENDELSE

  ENDELSE

END
