;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_EXTRACT_EDB
;
; Purpose     :	Extracts EDB data from science data stream.
;
; Explanation : Extracts EDB data from science data stream.
;
; Use         : < rcs_extract_edb, ids, data, headers, markers, indices, nmarkers, previous >
;
; Inputs      : ids      : STRUCTURE containg widget ids.
;               data     : BYTARR data stream.
;               headers  : STRUCTURE array containing header info for each packet.
;               markers  : LONARR start indices of EDBs within data.
;               indices  : LONARR indices within headers coresponding to EDBs.
;               nmarkers : LONG no. of markers.
;               previous : STRUCTURE containing info about previous packet.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, 13/12/00.
;                 Added silent check.
;               Version 0.2, 10/2/01
;                 Changed to silent routine.
;                 Changed edb_count and use to COMMON blocks.
;               Version 0.3, 14/03/01
;                 Removed ids from rcs_text, rcs_get_time, rcs_info and rcs_process_edb argument list.
;               Version 0.4, 04/04/01
;                 For synchronization warning added check on previous.pos to test whether just starting.
;                 Changed way last marker is processed.
;                 Changed test for dummy packet from edb_size to 512 bytes.
;               Version 0.5, 25/05/01
;                 Moved processing of time to avoid problem when partial packet updates tmax.
;               Version 0.6, 18/07/01
;                 Added CDROM version number.
;               Version 0.7, 02/09/03
;                 Changed version to source and cd_version.
;                 Changed format of vs to sss_v_a_b (previously sv)
;               Version 0.8, 08/01/04
;                 Renamed rcs sw version.
;
; Version     : Version 0.8, 08/01/04
;-
;**********************************************************

PRO rcs_extract_edb, ids, data, headers, markers, indices, nmarkers, previous

  ; get current count of EDBs

  edb_count = rcs_get_edb_count()

  ; get IF data required

  use = rcs_get_use()

  ; extract edb data

  FOR k = 0, nmarkers-1 DO BEGIN

    ; check edb info complete else leave
    ; if have complete sync marker and subcommutation index
    ; then output warnings about data lost or subcommutation index missing

    IF markers[k]+3 LT N_ELEMENTS(data) THEN BEGIN

      ; set warning flag off

      warning = 0

      ; get edb type

      type = rcs_get_type( data[markers[k]+2], headers[indices[k]] )

      ; get expected size of EDB

      edb_size = ([512,2304,512,2340])[type]

      ; get subcommutation indices

      subc = data[markers[k]+3]

      ; check if any data lost since last EDB

      IF markers[k]-previous.pos GT 0 THEN BEGIN

        ; allow extra dummy packet or initialization of previous.pos

        IF (previous.subc GE 0) AND (markers[k]-previous.pos NE 512)  THEN BEGIN

          warning = 1
          rcs_text, 'WARNING, SYNCHRONIZATION LOST, DATA GAP : '+$
                          rcs_strtrim(markers[k]-previous.pos)

        ENDIF

        ; update position

        previous.pos = markers[k]

      ENDIF

      ; check if OK to process EDB

      ok = 1

      IF k EQ nmarkers-1 THEN BEGIN

        ; check if last marker and insufficient data to see start of next marker else process later

        IF markers[k]+edb_size+1 GT N_ELEMENTS(data) THEN ok = 0

      ENDIF ELSE IF markers[k]+edb_size GT markers[k+1] THEN BEGIN

        ; insufficient data to next marker

        warning = 1

        rcs_text, 'WARNING, SYNCHRONIZATION LOST, INSUFFICIENT DATA : '+$
                         rcs_strtrim(markers[k+1]-markers[k])

        ; update position

        previous.pos = markers[k+1]

        ; dont process

        ok = 0

      ENDIF

      ; check if want to process this data

      IF ok THEN BEGIN

        ; have sufficient data to process this marker

        ; get edb data

        edb = data[markers[k]:markers[k]+edb_size-1]

        ; update position

        previous.pos = markers[k] + edb_size

        ; check if next EDB follows consecutively
        ; NB if last marker and get here then know have gap

        gap_error = 0
        IF k EQ (nmarkers-1) THEN gap_error = 1 ELSE $
          IF previous.pos NE markers[k+1] AND markers[k+1]-previous.pos NE 512 THEN gap_error = 1

        IF KEYWORD_SET(gap_error) THEN BEGIN

          rcs_text,'WARNING, GAP ERROR'

          warning=1

        ENDIF

        ; check subcommutation indices

        ; check first of the new subcommutation indices
        ; can only do this if previously found a complete EDB

        subc_invalid = 0

        IF previous.subc GE 0 THEN BEGIN

          IF subc NE ((previous.subc+1) MOD 256) THEN BEGIN
            warning = 1
            subc_invalid = 1
            rcs_text, 'WARNING, INVALID SUBCOMMUTATION INDEX'
          ENDIF

        ENDIF

        ; get character giving sc

        sc = headers[indices[k]].header.info.sc

        ; get version (sss_v_a_b)

        vs = rcs_if_version(SOURCE=headers[indices[k]].header.info.source, $
                            CD_VERSION=headers[indices[k]].header.info.cd_version, $
                            SW_VERSION=ies_sw_version() )

        ; get edb time

        time = rcs_get_time(headers, markers[k], indices[k])

        ; process edb

        rcs_process_edb, edb, time, type, subc, use, sc, vs, SUBC_INVALID=subc_invalid, GAP_ERROR=gap_error

        ; update counts

        CASE type OF
          0 : edb_count.nm  = edb_count.nm  + 1
          1 : edb_count.bm1 = edb_count.bm1 + 1
          2 : edb_count.bm2 = edb_count.bm2 + 1
          3 : edb_count.bm3 = edb_count.bm3 + 1
          ELSE : rcs_text, 'Invalid type encountered'
        ENDCASE

        ; update edb count

        rcs_set_edb_count, edb_count

        ; update state

        rcs_extract_state, ids

      ENDIF ELSE BEGIN

        ; not processing this marker

        ; dont update subc else wont report subc gap

        subc = previous.subc

        type = previous.type

      ENDELSE

      ; add info if warning

      IF warning THEN rcs_info, headers, indices[k], type, subc, previous.type, previous.subc

      ; store subcommutation index and type of next but last

      previous.subc = subc

      previous.type = type

    ENDIF

  ENDFOR

END
