;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_EXTRACT_HK
;
; Purpose     :	Extracts RAPID HK packets.
;
; Explanation : Extracts RAPID HK packets.
;
; Use         : < rcs_extract_hk, ids, headers, data, markers, nmarkers, previous >
;
; Inputs      : ids      : STRUCTURE containg widget ids.
;               headers  : STRUCTURE array containing header info for each packet.
;               data     : BYTARR data stream.
;               markers  : LONARR start indices of packets within data .
;               nmarkers : LONG no. of markers.
;               previous : STRUCTURE containing info about previous packet.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, 13/12/00.
;                 Added silent check.
;               Version 0.2, 24/1/01
;                 Modified check on time difference between HK frames to allow
;                 missing DAT file HK frames to get through.
;               Version 0.3, 10/2/01
;                 Moved silent to rcs_text.
;               Version 0.4, 14/03/01
;                 Removed ids from rcs_text, rcs_hk_info, rcs_check_hk argument list.
;               Version 0.5, 18/07/01
;                 Added CDROM version number.
;               Version 0.6, 02/09/03
;                 Changed version to source and cd_version.
;                 Changed format of vs to sss_v_a_b (previously sv)
;               Version 0.7, 08/01/04
;                 Renamed rcs sw version.
;
; Version     : Version 0.7, 08/01/04
;-
;**********************************************************

PRO rcs_extract_hk, ids, headers, data, markers, nmarkers, previous

  COMMON rcs_extract_block, tmax, data_reset

  ; get current count of EDBs

  edb_count = rcs_get_edb_count()

  ; get if data required

  use = rcs_get_use()

  ; extract hk packet

  FOR k = 0, nmarkers-1 DO BEGIN

    ; set warning flag off

    warning = 0

    ; get HK frame counter

    fc = data[markers[k]] AND '1F'XB

    ; check HK frame counters
    ; can only do this if have already found a complete EDB

    IF previous.fc GE 0 THEN IF fc NE ((previous.fc+1) MOD 32) THEN BEGIN
      warning = 1
      fc_invalid = 1
      rcs_text, 'INVALID HK FRAME COUNTER'
    ENDIF

    ; get HK packet time

    time = headers[k].header.time

    ; check time difference between successive HK frames
    ; NB nominal time difference is 5 secs
    ; For data from DAT files frame times are resampled to give
    ; HK frames every spin ie 4 secs.
    ; This results in every fourth HK frame being duplicated.
    ; This duplicate is removed from the HK stream within rcs_in.
    ; This results in a time gap of 8 secs between some HK frames
    ; when processing DAT files.
    ; Have changed upper time limit to 8.5 from 8 seconds to prevent
    ; warnings for this.

    IF previous.fc GE 0 THEN BEGIN

      ; NB do not set warning flag

      dt = time-previous.time

      IF dt LT 3 OR dt GT 8.5 THEN BEGIN

        rcs_text, 'WARNING, INVALID HK HEADER BLOCK TIME INTERVAL : '+STRING(dt)

      ENDIF

    ENDIF

    ; get sc

    sc = headers[k].header.info.sc

    ; get version (sss_v_a_b)

    vs = rcs_if_version(SOURCE=headers[k].header.info.source, $
                        CD_VERSION=headers[k].header.info.cd_version, $
                        SW_VERSION=ies_sw_version() )

    ; get edb data

    edb = data[markers[k]:markers[k]+40-1]

    ; get lut_no

    dummy = rcs_check_hk(edb, lut_no)

    ; get flag

    IF KEYWORD_SET(fc_invalid) THEN flag = '20'X ELSE flag = '00'X

    ; rcs_test_hk, edb

    ; check valid lut

    IF lut_no EQ -1 THEN warning = 1

    IF KEYWORD_SET(use.ehk) THEN BEGIN

      ; extract EHK [detectors]

      ehk = REFORM(rcs_decompress(edb[27:35]))

      ; add to common
      ; NB pass hkfcr by argument

      rcs_add_ehk, sc, vs, time, ehk, flag, lut_no, edb[0]

    ENDIF

    IF KEYWORD_SET(use.hk1) THEN rcs_add_hk1, sc, vs, time, edb, flag, lut_no

    IF KEYWORD_SET(use.hk8) THEN rcs_add_hk8, sc, vs, time, edb, flag

    IF KEYWORD_SET(use.hk32) THEN rcs_add_hk32, sc, vs, time, edb, flag

    ; store previous

    previous.fc = fc

    previous.time = time

    ; update state

    rcs_extract_state, ids, /HK

    ; add info if warning

    IF warning THEN rcs_hk_info, headers[k], fc, previous.fc

    ; update counts

    edb_count.hk = edb_count.hk + 1

    rcs_set_edb_count, edb_count

  ENDFOR

  ; store latest time

  tmax = time

END
