;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_IFDATA
;
; Purpose     :	Gets IF data spanning a specified period.
;
; Explanation : Gets IF data spanning a specified period.
;               IF data is stored in daily files.
;               There is a complication that IF data generated from DDS (SCI ?) files
;               may contain a few seconds worth of data from the previous day and may be missing a few
;               seconds worth of data from the end of the day.
;
; Use         : < res = rcs_get_ifdata(sc, datatype, t0, t1, input_header, input_data) >
;
; Inputs      : sc         : STRING character giving SC
;               datatype   : STRING giving data type
;               start_time : DOUBLE TAI start time
;               end_time   : DOUBLE TAI stop time
;
; Opt. Inputs : None.
;
; Outputs     : res          : INT : 0=failed, 1=ok
;               input_header : STRUCTURE
;               input_data   : STRUCTURE
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 15/02/02
;
; Modified    : Version 0.1, 13/03/02
;                 Changed sc argument to sc_root in rcs_restore_ifdata.
;               Version 0.2, 09/04/02
;                 Fixed bug with missing data at end of day.
;               Version 0.3, 21/09/03
;                 Removed attempt to grab next day of data to cope with missing data from end of day behaviour.
;
; Version     :	Version 0.3, 21/09/03
;-
;**********************************************************

FUNCTION rcs_get_ifdata, sc, datatype, start_time, end_time, header, data, QUIET=quiet

  ; store one day of IF data in case needed again

  COMMON rcs_get_ifdata_local, xscname, xdatatype, xday, res, input_header, input_data

  IF N_ELEMENTS(xscname) EQ 0 THEN BEGIN

    xscname = ''
    xdatatype = ''
    xday = ''

  ENDIF

  ; mjd is 00:00:00 on day containing start time

  mjd = anytim2utc(start_time)

  mjd.time = 0

  ; set up TAI data end time

  endt = anytim2tai(end_time)

  header = 0

  data = 0

  ; data in daily files does not quite span 24 hour period
  ; ie 22/Nov/01 23:59:31 23/Nov/01 23:59:23
  ; (at least from DDS)
  ; Previously tried to avoid missing off small chunk at end get next file
  ; if within 1 minute of end
  ; (checked first though that have not got all data needed anyway
  ;  by seeing if current data end time within 6 seconds of plot end
  ;  keep local copy in common block to avoid retrieving days twice
  ;  NB this creates a potential problem if different versions of data processed for the same day
  ;  will not force retrieval of new data from file)

  ; data_endt = 0.0d0
  ; WHILE (endt-data_endt) GT 6 AND anytim2tai(mjd) LT endt+60 DO BEGIN
  ; data_endt = input_header.tstop

  ; Removed this since dds complications without much improvement

  WHILE anytim2tai(mjd) LT endt DO BEGIN

    day = rcs_get_yymmdd(mjd)

    scname = rcs_get_scname(sc)

    IF NOT (scname EQ xscname AND day EQ xday AND datatype EQ xdatatype) THEN BEGIN

      res = rcs_restore_ifdata(scname, datatype, day, input_header, input_data, QUIET=quiet)

      xscname = scname

      xdatatype = datatype

      xday = day

    ENDIF ELSE BEGIN

      IF NOT KEYWORD_SET(quiet) THEN rcs_text, 'USING BUFFERED IF DATA SET FOR : '+scname+' '+datatype+' '+day, /OVERRIDE

    ENDELSE

    IF KEYWORD_SET(res) THEN BEGIN

      rcs_concatenate, header, data, input_header, input_data

    ENDIF

    mjd.mjd = mjd.mjd + 1

  ENDWHILE

  IF KEYWORD_SET(header) THEN RETURN, 1 ELSE RETURN, 0

END
