;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_IFFILES
;
; Purpose     :	Gets filenames of all IF files for a particular spacecraft.
;
; Explanation : Gets filenames of all IF files for a particular spacecraft.
;               Includes path.
;               Returns only files from specified source.
;               NB by setting source='' and sw_version='' can search for files yymmdd*.dat.
;
; Use         : < ifnames = rcs_get_iffiles(sc) >
;
; Inputs      : sc : STRING character giving S/C
;
; Opt. Inputs : None.
;
; Outputs     : ifnames : STRARR all IF files.
;
; Opt. Outputs: None.
;
; Keywords    : COUNT      : INT returns no. of if files
;               DAY        : STRING day filter : yymmdd
;               SOURCE     : STRING source filter
;               CD_VERSION : STRING cd version ie 'a'
;               SW_VERSION : STRING sw version ie 2_3
;
; Written     :	Version 0.0, 22/03/01
;
; Modified    : Version 0.1, 27/07/01
;                 Replaced FILTER keyword with DAY and SOURCE.
;               Version 0.2, 09/04/02
;                 Added sw version to filter.
;               Version 0.3, 04/07/02
;                 Small changes to comments.
;               Version 0.4, 08/08/02
;                 Added source version to filter.
;               Version 0.5, 13/02/03
;                 Added sw version keyword.
;               Version 0.6, 03/04/03
;                 Used rcs get dir.
;               Version 0.7, 02/09/03
;                 Changed source IDs.
;                 Added CD_VERSION keyword.
;                 Changed so that should not use combined source and cd_version.
;                 Set up default for sw_version and source.
;               Version 0.8, 07/11/03
;                 Changed rcs get ifdir to rcs get subdir.
;
; Version     :	Version 0.8, 07/11/03
;-
;**********************************************************

FUNCTION rcs_get_iffiles, sc, COUNT=ifcount, DAY=day, SOURCE=source, CD_VERSION=cd_version, SW_VERSION=sw_version

  ; initialize

  iffiles = ''

  ifcount = 0

  ; change default for source and sw_version

  IF N_ELEMENTS(source) EQ 0 THEN source = rcs_get_source()

  IF N_ELEMENTS(sw_version) EQ 0 THEN sw_version = rcs_get_sw_version()

  ; allow yymmdd.dat searches

  filter = rcs_if_filter(DAY=day, SOURCE=source, CD_VERSION=cd_version, SW_VERSION=sw_version)

  ; get directory containing if DATA set for this SC
  ; (will create if necessary)

  path = rcs_get_subdir('IF', sc)

  ; check directory exists

  IF NOT rcs_dir_exist(path) THEN RETURN, iffiles

  ; get all directory names below this
  ; NB this would include subdirectories if there were any

  dirs = rcs_find_subdirs(path)

  ; loop through these extracting filenames

  FOR k = 0, N_ELEMENTS(dirs)-1 DO BEGIN

    files = FINDFILE(concat_dir(dirs[k],filter),COUNT=count)

    IF count GT 0 THEN iffiles = [iffiles, files]

  ENDFOR

  ifcount = N_ELEMENTS(iffiles)-1

  IF ifcount GT 0 THEN iffiles = iffiles[1:*]

  IF ifcount GT 0 AND source EQ '' THEN BEGIN

    ; return only yymmdd.dat files

    list = WHERE(STRLEN(strip_dirname(iffiles)) EQ 10, ifcount)

    IF ifcount GT 0 THEN iffiles = iffiles[list] ELSE iffiles = ''

  ENDIF

  RETURN, iffiles

END
