;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	rcs_get_luts
;
; Purpose     :	Define number 'energy' channels, 'detector' channels, 'sectors', 'luts', 'changes' for each IF
;               data set.
;
; Explanation : These terms are defined in terms of IES spectra but are adapted for other data sets in a
;               data set specific manner.
;               For IES :
;               The bin positions and sizes of the energy channels differ from detector to detector.
;               The bin positions of the energy channels for each detector are stored in uploadable lookup tables.
;               Separate lookup tables exist for different integration times.
;               The lookup tables have occasionally changed.
;               The energy bands of the data products are selected from the energy channels.
;               The channels selected have occasionally changed.
;
;               Other calibration tables are also set-up within this routine.
;
; Use         : <luts = rcs_get_luts (sc, data_type, lut_times, lut_map, count_calibration, energy_calibration, $
;                                     energy_efficiency)>
;
; Inputs      : sc        : STRING character giving spacecraft ID
;               data_type : STRING giving IF data set
;
; Opt. Inputs : None.
;
; Outputs     : lut_times          : STRARR(no_of_changes) time after which LUT valid.
;               lut_map            : INTARR(no_of_luts) integration mode for each LUT
;               luts               : BYTARR (2, no_of_energies, no_of_detectors, no_of_luts)
;                                          top and bottom bin positions of energy channels for each detector,
;               count_calibration  : FLTARR(no_of_luts) calibration factor for each look up table.
;               energy_calibration : FLTARR[1] or FLTARR[2] calibration factor for bins to energy.
;                                          may have energy offset also.
;              	energy_efficiency  : FLTARR(no_of_energies,no_of_luts)
;                                          efficiency factor for each energy and look up table.
;
; Opt. Outputs:	NO_OF_ENERGIES  : INT named variable
;               NO_OF_DETECTORS : INT named variable
;               NO_OF_LUTS      : INT named variable
;               NO_OF_CHANGES   : INT named variable
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/9/00
;
; Modified    :	Version 0.1, 26/1/01
;                 Added 'E3DDF' data type.
;           	Version 0.2, 3/2/01
;                 Corrected bug for CLEAN if more than one lut_time.
;               Version 0.3, 6/2/01
;                 Added pedestal and espct IF data
;               Version 0.4, 08/04/01
;                 Corrected geometric factor for ESPCT (divide by 9).
;                 Corrected geometric factor for I3DD (multiply by 12)
;                 Put in energy efficiiencies for.HSPCT and ISPCT.
;               Version 0.5, 15/05/01
;                 Added CLEANBM2.
;               Version 0.6, 25/05/01
;                 Added extra items to HK1 and HK32.
;               Version 0.7, 08/02/02
;                 Added special case for EPAD data where bin definition changed.
;                 Added EPAD2
;               Version 0.8, 20/06/02
;                 Corrected energy_efficiency for PEDESTAL.
;               Version 0.9, 15/11/02
;                 Used ntimes in calculations.
;                 Added new energy channel definitions for version 0.3 luts.
;                 Affects EPAD, EPAD2.
;                 Corrected EPAD and EPAD2 bin size if high energy channel.
;               Version 1.0, 21/11/02
;                 Used rcs_ies_luts.
;               Version 1.1, 06/02/03
;                 Corrected bug in EHIST code introduced in 1.0
;               Version 1.2, 19/02/03
;                 Corrected 2nd bug in EHIST code introduced in 1.0
;                 Forgot to use /HIST keyword in call to ies_get_luts.
;               Version 1.3, 06/03/03
;                 When switching to use rcs_ies_luts forgot that ESPCT has different count calibration factors.
;               Version 1.4, 01/12/03
;                 Changed around argument list.
;
; Version     :	Version 1.4, 01/12/03
;-
;**********************************************************

FUNCTION rcs_get_luts, sc, data_type, lut_times, lut_map, count_calibration, energy_calibration, energy_efficiency, $
         NO_OF_ENERGIES=no_of_energies, NO_OF_DETECTORS=no_of_detectors, NO_OF_LUTS=no_of_luts, NO_OF_CHANGES=no_of_changes

  ; set up
  ; lut_times          : STRARR(no_of_changes)
  ; luts               : BYTARR (2, no_of_energies, no_of_detectors, no_of_luts)
  ; count_calibration  : FLTARR(no_of_luts)
  ; energy_calibration : FLTARR[1] or FLTARR[2]
  ; energy_efficiency  : FLTARR(no_of_energies,no_of_luts)
  ; lut_map            : INTARR(no_of_luts)

  ; on board have 16 energy channels for IES (except for Histogram mode)
  ; energy bands for the various products are derived from these energy channels
  ; the eneryg channel selections have occasionally changed.

  ; set up luts array (such that 1-0 is size)
  ; 1st dimension gives start and end of each energy band
  ; 2nd dimension gives detector
  ; 3rd dimension gives integration mode for each LUT change

  ; do energy channel selection

  CASE data_type OF

    'E3DD' : BEGIN

                 ; NM energy bands

                 rcs_ies_luts, sc, 8, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,8,9,4*N_ELEMENTS(lut_times))

                 luts[0,*,*,0:7] = lut_boundaries[[3,4,6,7,8,9,10,12],*,0:7]

                 luts[1,*,*,0:7] = lut_boundaries[[4,5,7,8,9,10,12,14],*,0:7]

                 luts[0,*,*,8:11] = lut_boundaries[[2,4,6,7,8,9,10,12],*,8:11]

                 luts[1,*,*,8:11] = lut_boundaries[[4,5,7,8,9,10,12,14],*,8:11]

               END

    'E3DDF' : BEGIN

                 ; full 16 energy channels

                 rcs_ies_luts, sc, 16, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,16,9,4*N_ELEMENTS(lut_times))

                 luts[0,*,*,*] = lut_boundaries[0:15,*,*]

                 luts[1,*,*,*] = lut_boundaries[1:16,*,*]

               END

    'E3DDBM' : BEGIN

                 ; BM energy bands

                 rcs_ies_luts, sc, 12, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,12,9,4*N_ELEMENTS(lut_times))

                 luts[0,*,*,*] = lut_boundaries[[2,3,4,5,6,7,8,9,10,11,12,13],*,*]

                 luts[1,*,*,*] = lut_boundaries[[3,4,5,6,7,8,9,10,11,12,13,14],*,*]

               END

    'CLEAN' : BEGIN

                 ; CLEAN energy bands

                 rcs_ies_luts, sc, 6, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,6,9,4*N_ELEMENTS(lut_times))

               END

    'ESPCT' : BEGIN

                 ; ESPCT energy bands

                 rcs_ies_luts, sc, 6, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 ; adjust count calibration to reflect different geometric factor of the 9 detectors taken together

                 count_calibration = count_calibration/9.0

                 luts = FLTARR(2,6,1,4*N_ELEMENTS(lut_times))

               END

    'CLEANBM' : BEGIN

                 ; CLEANBM energy bands

                 rcs_ies_luts, sc, 8, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,8,9,4*N_ELEMENTS(lut_times))

               END

    'CLEANBM2' : BEGIN

                 ; CLEANBM2 energy bands

                 rcs_ies_luts, sc, 1, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,1,9,4*N_ELEMENTS(lut_times))

               END

    'PEDESTAL' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)
               ; arrange as 9 detectors with 5 energy bands

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,5,9,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 5, 4*N_ELEMENTS(lut_times))

             END

    'EPAD' : BEGIN

                 ; EPAD energy bands

                 rcs_ies_luts, sc, 2, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,2,9,4*N_ELEMENTS(lut_times))

                 luts[0,*,*,0:3] = lut_boundaries[[6,10],*,0:3]

                 luts[1,*,*,0:3] = lut_boundaries[[10,14],*,0:3]

                 luts[0,*,*,4:7] = lut_boundaries[[7,10],*,4:7]

                 luts[1,*,*,4:7] = lut_boundaries[[10,14],*,4:7]

                 luts[0,*,*,8:11] = lut_boundaries[[6,8],*,8:11]

                 luts[1,*,*,8:11] = lut_boundaries[[8,10],*,8:11]

               END

    'EPAD2' : BEGIN

                 ; EPAD2 energy bands

                 rcs_ies_luts, sc, 2, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,2,9,4*N_ELEMENTS(lut_times))

                 luts[0,*,*,0:3] = lut_boundaries[[6,10],*,0:3]

                 luts[1,*,*,0:3] = lut_boundaries[[10,14],*,0:3]

                 luts[0,*,*,4:7] = lut_boundaries[[7,10],*,4:7]

                 luts[1,*,*,4:7] = lut_boundaries[[10,14],*,4:7]

                 luts[0,*,*,8:11] = lut_boundaries[[6,8],*,8:11]

                 luts[1,*,*,8:11] = lut_boundaries[[8,10],*,8:11]

               END

    'EHK' : BEGIN

                 ; EHK energy bands

                 rcs_ies_luts, sc, 4, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times

                 luts = FLTARR(2,4,9,4*N_ELEMENTS(lut_times))

                 luts[0,*,*,0:3] = lut_boundaries[[0,1,14,15],*,0:3]

                 luts[1,*,*,0:3] = lut_boundaries[[1,2,15,16],*,0:3]

                 luts[0,*,*,4:7] = lut_boundaries[[0,7,10,14],*,4:7]

                 luts[1,*,*,4:7] = lut_boundaries[[2,10,14,16],*,4:7]

                 luts[0,*,*,8:11] = lut_boundaries[[0,2,14,15],*,8:11]

                 luts[1,*,*,8:11] = lut_boundaries[[2,6,15,16],*,8:11]

               END

    'EHIST' : BEGIN

                 ; store information about detector in lut no changes to save space
                 ; luts = FLTARR(2,nbands,1,nluts, ndetectors)
                 ; NB count_calibration is indexed by adjusted lut_no
                 ; NB require EHIST not to have any changes because of indexing scheme used for detectors.

                 rcs_ies_luts, sc, 256, lut_boundaries, count_calibration, $
                  energy_calibration, energy_efficiency, lut_map, lut_times, /HIST

                 luts = FLTARR(2,256,1,4*9)

                 luts[0,*,*,*] = FINDGEN(256)#REPLICATE(1,4*9)

                 luts[1,*,*,*] = luts[0,*,*,*] + 1

               END

    'I3DD_H' : BEGIN

                 ; luts = FLTARR(2,nbands,ndetectors,nluts)

                 lut_times = anytim2tai(['1/Jan/1990'])

                 luts = FLTARR(2,8,12,1*N_ELEMENTS(lut_times))

                 energies = [30.3,67.7,95.1,168.6,410.3,1107,3055,27500,200000]   ; keV

                 luts[0,*,*,0] = energies[0:7]#REPLICATE(1,12)
                 luts[1,*,*,0] = energies[1:8]#REPLICATE(1,12)

                 count_calibration = [1.]*12.

                 ; set up energy calibration as [slope (keV/ADC), offset (keV)]

                 energy_calibration = [1.0]

                 energy_efficiency = [0.00108,1.69,2.02,2.17,2.06,1.95,1.95,1.95] # REPLICATE(1.0, N_ELEMENTS(lut_times))

               END

    'I3DD_He' : BEGIN

                 ; luts = FLTARR(2,nbands,ndetectors,nluts)

                 lut_times = anytim2tai(['1/Jan/1990'])

                 luts = FLTARR(2,8,12,1*N_ELEMENTS(lut_times))

                 energies = [100.1,139.2,175.5,241.7,366.5,787.1,1920,4427,20200]   ; keV

                 luts[0,*,*,0] = energies[0:7]#REPLICATE(1,12)
                 luts[1,*,*,0] = energies[1:8]#REPLICATE(1,12)

                 count_calibration = [1.]*12.

                 ; set up energy calibration as [slope (keV/ADC), offset (keV)]

                 energy_calibration = [1.0]

                 energy_efficiency = [0.00193,2.44,2.94,3.5,3.93,3.42,3.42,3.42] # REPLICATE(1.0, N_ELEMENTS(lut_times))

               END

    'I3DD_CNO' : BEGIN

                 ; luts = FLTARR(2,nbands,ndetectors,nluts)

                 lut_times = anytim2tai(['1/Jan/1990'])

                 luts = FLTARR(2,8,12,1*N_ELEMENTS(lut_times))

                 energies = [105.1,302,463.8,562.7,730.5,1105,1680,3557,8970]   ; keV

                 luts[0,*,*,0,0] = energies[0:7]#REPLICATE(1,12)
                 luts[1,*,*,0,0] = energies[1:8]#REPLICATE(1,12)

                 count_calibration = [1.]*12.

                 ; set up energy calibration as [slope (keV/ADC), offset (keV)]

                 energy_calibration = [1.0]

                 energy_efficiency = [0.00130,2.36,2.78,3.04,3.22,3.22,3.22,3.22] # REPLICATE(1.0, N_ELEMENTS(lut_times))

               END

    'IPAD' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,2,12,1*N_ELEMENTS(lut_times))

               luts[0,*,*,0] = FINDGEN(2)#REPLICATE(1,12)
               luts[1,*,*,0] = (1+FINDGEN(2))#REPLICATE(1,12)

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 2, N_ELEMENTS(lut_times))

             END

    'HSPCT' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,8,1,1*N_ELEMENTS(lut_times))

               energies_h = [30.3,67.7,95.1,168.6,410.3,1107,3055,27500,200000]   ; keV

               luts[0,*,0,0] = energies_h[0:7]
               luts[1,*,0,0] = energies_h[1:8]

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = [0.00108,1.69,2.02,2.17,2.06,1.95,1.95,1.95] # REPLICATE(1.0,N_ELEMENTS(lut_times))

             END

    'ISPCT_He' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,8,1,1*N_ELEMENTS(lut_times))

               energies = [100.1,139.2,175.5,241.7,366.5,787.1,1920,4427,20200]   ; keV

               luts[0,*,0,0] = energies[0:7]
               luts[1,*,0,0] = energies[1:8]

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = [0.00193,2.44,2.94,3.5,3.93,3.42,3.42,3.42] # REPLICATE(1.0, N_ELEMENTS(lut_times))

             END

    'ISPCT_CNO' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,8,1,1*N_ELEMENTS(lut_times))

               energies= [105.1,302,463.8,562.7,730.5,1105,1680,3557,8970]   ; keV

               luts[0,*,0,0] = energies[0:7]
               luts[1,*,0,0] = energies[1:8]

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = [0.00130,2.36,2.78,3.04,3.22,3.22,3.22,3.22] # REPLICATE(1.0, N_ELEMENTS(lut_times))

             END

    'MTRX' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)
               ; arrange A-E/A matrix as A=32 mass classes=sectors,
               ; E/A=64 energy/mass classes=detectors

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,64,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

    'SGLBM' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)
               ; arrange as 5 detectors with 1 energy band

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,5,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

    'SGL01' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)
               ; arrange as 7 detectors with 1 energy band

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,7,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

    'SGL2' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)
               ; arrange as 18 detectors with 1 energy band

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,18,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

    'SGL3' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)
               ; arrange as 22 detectors with 1 energy band

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,22,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

     'HK1' : BEGIN

               ; detectors=['ERISTAHV','ERISTOHV','ERIDEFHV','ERDTRIGM','ERDIFIND','ERDIFCAL','ERDDEADT','ERDHKFCR',
               ;            'ERISTOCB','ERISTOCB','ERIENYCB','ERDTMMOD']

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,12,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

     'HK2' : BEGIN

               ; not set up yet
               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,8,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

     'HK4' : BEGIN

               ; not set up yet

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,8,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

     'HK8' : BEGIN

               ; 'ERDGNDRF','ERIP5VRF','ERIM5VRF','ERIP12RF','ERIM12RF','ERISAREF','ERISTREF','ERIHKTRF'

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,8,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

     'HK32' : BEGIN

               ; 'ERDTOERC','ERDFRPRT','EREFXLUT'

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,5,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0,0] = 0
               luts[1,0,*,0,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

     'SCINFO' : BEGIN

               ; luts = FLTARR(2,nbands,ndetectors,nluts)

               lut_times = anytim2tai(['1/Jan/1990'])

               luts = FLTARR(2,1,2,1*N_ELEMENTS(lut_times))

               luts[0,0,*,0] = 0
               luts[1,0,*,0] = 1

               count_calibration = [1.0]

               ; set up energy calibration as [slope (keV/ADC), offset (keV)]

               energy_calibration = [1.0]

               energy_efficiency = REPLICATE(1.0, 1, N_ELEMENTS(lut_times))

             END

     ELSE : rcs_message, 'INVALID DATA TYPE : '+data_type

  ENDCASE

  ; get no of energies for experiment

  IF (SIZE(luts))[0] GT 1 THEN no_of_energies = (SIZE(luts))[2] ELSE no_of_energies = 1

  ; get no. of detectors

  IF (SIZE(luts))[0] GT 2 THEN no_of_detectors = (SIZE(luts))[3] ELSE no_of_detectors = 1

  ; get no. of LUTs

  IF (SIZE(luts))[0] GT 3 THEN no_of_luts = (SIZE(luts))[4] ELSE no_of_luts = 1

  ; get no. of LUT changes

  no_of_changes = N_ELEMENTS(lut_times)

  RETURN, luts

END
