;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_TIME
;
; Purpose     :	Calculates time of EDB.
;
; Explanation : Uses time of neghbouring packet headers and interpolation
;               within packet to get time of EDB.
;
; Use         : < time = rcs_get_time(headers, marker, index, WARNING=warning) >
;
; Inputs      : headers  : STRUCTURE array containing header info for each packet.
;               marker   : LONG start index of packets within data .
;               index    : LONG index of packet header corresponding to
;                          packet containing start of EDB.
;
; Opt. Inputs : None.
;
; Outputs     : time     : DOUBLE TAI time of start of EDB.
;
; Opt. Outputs: None.
;
; Keywords    : WARNING : LONG returns error warning.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, 13/12/00
;                 Added SILENT check.
;               Version 0.2, 12/2/01
;                 Removed silent keyword.
;               Version 0.3, 14/03/01
;                 Removed ids from rcs_text and this argument list.
;
; Version     :	Version 0.3, 14/03/01
;-
;**********************************************************

FUNCTION rcs_get_time, headers, marker, index, WARNING=warning

  COMMON rcs_extract_block, tmax, data_reset

  ; estimate times of EDB

  ; get current block time and next block time
  ; check if last block

  IF index LT N_ELEMENTS(headers)-1 THEN BEGIN

    next_block = headers[index+1]

    ; get current block time

    current_block = headers[index]

    ; if dt sensible then interpolate between the two

    dt = next_block.header.time - current_block.header.time

    IF dt GE 3 AND dt LE 8 THEN BEGIN

      time = current_block.header.time +  dt*(marker-current_block.index) $
                                            /current_block.header.length
    ENDIF ELSE BEGIN

      warning = 1

      rcs_text, 'WARNING, INVALID HEADER BLOCK TIME INTERVAL : '+STRING(dt)

      ; use default block time and block size

      time = current_block.header.time +  5.12*(marker-current_block.index) $
                                            /current_block.header.length

    ENDELSE

  ENDIF ELSE BEGIN

    ; last header block : no next block time

    current_block = headers[index]

    ; use default block time and block size

    time = current_block.header.time +  5.12*(marker-current_block.index) $
                                          /current_block.header.length

  ENDELSE

  ; store latest time

  tmax = time

  RETURN, time

END
