;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	rcs_ies_luts
;
; Purpose     :	Define IES lookup tables.
;
; Explanation : The bin positions and sizes of the energy channels differ from detector to detector.
;               The bin positions of the energy channels for each detector are stored in uploadable lookup tables.
;               Separate lookup tables exist for different integration times.
;               The lookup tables have occasionally changed.
;               This routine returns the lookup tables.
;               The new lookup tables are appended to the existing ones.
;               Associated with the lookup tables will be a calibration table giving
;               a calibration factor for each table. This normalizes the data taken with each table
;               to counts per cm2 per sr (detector deadtime removed for IES).
;               Other calibration tables are also set-up within this routine.
;
; Use         : <rcs_ies_luts, sc, nbands, luts, count_calibration, energy_calibration, $
;                                            energy_efficiency, lut_map, lut_times)>
;
; Inputs      : sc          : STRING character giving spacecraft ID
;               nbands      : INT no of energy bands
;
; Opt. Inputs : None.
;
; Outputs     : luts               : BYTARR (2, no_of_energies, no_of_detectors, no_of_luts)
;                                          top and bottom bin positions of energy channels for each detector,
;               count_calibration  : FLTARR(no_of_luts) calibration factor for each look up table.
;               energy_calibration : FLTARR[1] or FLTARR[2] calibration factor for bins to energy.
;                                          may have energy offset also.
;              	energy_efficiency  : FLTARR(no_of_energies,no_of_luts)
;                                          efficiency factor for each energy and look up table.
;               lut_map            : INTARR(no_of_luts) integration mode for each LUT
;               lut_times          : STRARR(no_of_changes) time after which LUT valid.
;
; Opt. Outputs:	None.
;
; Keywords    : HIST : flag indicating histogram data..
;
; Written     :	Version 0.0, Martin Carter, RAL, 21/11/02
;                 Adapted from rcs get luts.
;
; Modified    : Version 0.1, 06/02/03
;                 Added HIST flag.
;               Version 0.2, 09/12/03
;                 Changed so that lut_map is repeated for each lut value.
;
; Version     :	Version 0.2, 09/12/03
;-
;**********************************************************

PRO rcs_ies_luts, sc, nbands, luts, count_calibration, energy_calibration, energy_efficiency, lut_map, lut_times, HIST=hist

  ; calculate conversion factors for cts --> cts/s/cm2/sr

  integration_times = [2.,5.,15.,50.] ; us

  deadtime = 50.0 ; us

  geometric_factor = 0.0022 ; cm2sr

  IF NOT KEYWORD_SET(hist) THEN BEGIN

    ; set up calendar dates of lut changes
    ; lut_starts : [no_of_energies+1,no_of_detectors,no_of_luts]

    luts = rcs_calculate_luts (  sc, lut_times)

    count_calibration = (deadtime + integration_times)/integration_times/geometric_factor # REPLICATE(1,N_ELEMENTS(lut_times))

  ENDIF ELSE BEGIN

    lut_times = anytim2tai(['1/Jan/1990'])

    count_calibration = (deadtime + integration_times)/integration_times/geometric_factor # REPLICATE(1,9)

  ENDELSE

  count_calibration = REFORM(count_calibration, N_ELEMENTS(count_calibration))

  ; set up energy calibration as [slope (keV/ADC), offset (keV)]

  energy_calibration = [2.2,6.5]

  ; set up lut_map
  ; maps lut no to integration modes
  ; NB for CEPPAD have lut_map[12,3] = [ntables,nchanges]
  ;    and lut_map = 0-->7 ie 2S 2L 10S 10L 50S 50L 100S 100L
  ;    and use lut_map MOD 2 EQ 0 to indicate small pixel values;
  ;    In order to cope with lut=-1 have removed dependence on
  ;    this indexing.
  ;    Also used for indexing into pedestals array i.e FLTARR[9,256,8,4]
  ;    with 4 temperatures, 8 modes.
  ;    In order to cope with lut=-1 have indexed lut_map with lut GT 0.
  ; Use lut_map = [0,1,2,3] for 2,5,15,50.
  ;    09/12/03  changed so that lut list ensures lut > 0 and indexd lut_map with lut.
  ;              repeated lut_map for each lut change.

  lut_map = INDGEN(4)#REPLICATE(1,N_ELEMENTS(lut_times))

  energy_efficiency = REPLICATE(1.0, nbands, 4*N_ELEMENTS(lut_times))

END
