;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PEDESTAL_INFO
;
; Purpose     : Derives pedestal information and optionally subtracts pedestal from each sample.
;
; Explanation :	Derives pedestal information and optionally subtracts pedestal from each sample.
;
; Use         : < pedestals = rcs_pedestal_info (ped, pp, array, tot_exp_counts, pedestal_centre, channel_centres, pedestal_posn) >
;
; Inputs      : ped              : FLTARR[npeds] estimate of pedestal contribution over pedestal channels.
;               pp               : INT pedestal shift index
;               array            : FLTARR(nbands) containing input data for each energy channel
;                                  May be changed on output if SUB flag set.
;               tot_exp_counts   : LONG total no. of counts expected for integration time (if no compression).
;               pedestal_centre  : FLOAT position of center of pedestal channels.
;               channel_centres  : FLTARR(npeds) position of centers of pedestal channels.
;               pedestal_posn    : FLOAT result of fit to pedestal.
;
; Opt. Inputs : None.
;
; Outputs     : pedestals : FLTARR[5]
;                           0 : pedestal bin position measured by cross_correlation algorithm
;                           1 : pedestal shift from central bin for cross_correlation algorithm
;                           2 : pedestal shift from central bin for centroid algorithm
;                           3 : occupancy defined as the ratio of non-pedestal counts to total counts expressed as a percentage
;                           4 : visible counts defined as ratio of counts in data channels vs expected total counts
;                               expressed out of 10 as missing counts
;
; Opt. Outputs:	None.
;
; Keywords    :
;
; Written     :	Version 0.0, 26/11/03
;                 Separate from cep pedestals.
;
; Modified    : Version 0.1, 20/01/04
;                 Separated SUB calculation.
;
; Version     :	Version 0.1, 20/01/04
;-
;**********************************************************

FUNCTION rcs_pedestal_info, ped, pp, array, tot_exp_counts, pedestal_centre, channel_centres, pedestal_posn

  ; set up no. of pedestal channels

  npeds = N_ELEMENTS(ped)

  ; get totals

  tot_ped_array = TOTAL(array[0:npeds-1])

  tot_array = tot_ped_array + TOTAL(array[npeds:*])

  ; set up pedestal info

  pedestals = FLTARR(5)

  ; set up pedestal bin position

  pedestals[0] = pedestal_posn - (pp-50)/10.0

  ; set up pedestal bin shift

  pedestals[1] = pedestals[0] - pedestal_centre

  ; get centroid of pedestal channels

  IF tot_ped_array GT 0 THEN pedestals[2] = TOTAL( array[0:npeds-1]*channel_centres ) / $
                                                   tot_ped_array - pedestal_centre

  ; set up occupancy

  IF tot_array GT 0 THEN pedestals[3] =  100.0*(1 - tot_ped_array/tot_array)

  ; set up missing counts

  pedestals[4] =  100.0*(1-tot_array / tot_exp_counts)

  RETURN, pedestals

END