; $Id: rcs_curvefit.pro,v 1.2 2004/02/26 01:53:56 friedel Exp $
;
; Copyright (c) 1982-2001, Research Systems, Inc.  All rights reserved.
;	Unauthorized reproduction prohibited.
;
;+
; NAME:
;       CURVEFIT
;
; PURPOSE:
;       Non-linear least squares fit to a function of an arbitrary
;       number of parameters.  The function may be any non-linear
;       function.  If available, partial derivatives can be calculated by
;       the user function, else this routine will estimate partial derivatives
;       with a forward difference approximation.
;
; CATEGORY:
;       E2 - Curve and Surface Fitting.
;
; CALLING SEQUENCE:
;       Result = CURVEFIT(X, Y, Weights, A, FUNCTION_NAME = name)
;
; INPUTS:
;       X:  A row vector of independent variables.  This routine does
;           not manipulate or use values in X, it simply passes X
;           to the user-written function.
;
;       Y:  A row vector containing the dependent variable.
;
;  Weights:  A row vector of weights, the same length as Y.
;            For statistical (Poisson)  weighting,
;                 Weights(i) = 1.0/y(i), etc.
;
;       A:  A vector, with as many elements as the number of terms, that
;           contains the initial estimate for each parameter.  IF A is double-
;           precision, calculations are performed in double precision,
;           otherwise they are performed in single precision. Fitted parameters
;           are returned in A.
;
; KEYWORDS:
;       FUNCTION_NAME:  The name of the function (actually, a procedure) to
;       fit.  IF omitted, "FUNCT" is used. The procedure must be written as
;       described under RESTRICTIONS, below.
;
; OUTPUTS:
;       Returns a vector of calculated values.
;       A:  A vector of parameters containing fit.
;
; OPTIONAL OUTPUT PARAMETERS:
;
; COMMON BLOCKS:
;       NONE.
;
; SIDE EFFECTS:
;       None.
;
; RESTRICTIONS:
;       The function to be fit must be defined and called FUNCT,
;       unless the FUNCTION_NAME keyword is supplied.  This function,
;       (actually written as a procedure) must accept values of
;       X (the independent variable), and A (the fitted function's
;       parameter values), and return F (the function's value at
;       X), and PDER (a 2D array of partial derivatives).
;       For an example, see FUNCT in the IDL User's Libaray.
;       A call to FUNCT is entered as:
;       FUNCT, X, A, F, PDER
; where:
;       X = Variable passed into CURVEFIT.  It is the job of the user-written
;           function to interpret this variable.
;       A = Vector of NTERMS function parameters, input.
;       F = Vector of NPOINT values of function, y(i) = funct(x), output.
;       PDER = Array, (NPOINT, NTERMS), of partial derivatives of funct.
;               PDER(I,J) = DErivative of function at ith point with
;               respect to jth parameter.  Optional output parameter.
;               PDER should not be calculated IF the parameter is not
;               supplied in call. IF the /NODERIVATIVE keyword is set in the
;               call to CURVEFIT THEN the user routine will never need to
;               calculate PDER.
;
; PROCEDURE:
;       Copied from "CURFIT", least squares fit to a non-linear
;       function, pages 237-239, Bevington, Data Reduction and Error
;       Analysis for the Physical Sciences.  This is adapted from:
;       Marquardt, "An Algorithm for Least-Squares Estimation of Nonlinear
;       Parameters", J. Soc. Ind. Appl. Math., Vol 11, no. 2, pp. 431-441,
;       June, 1963.
;
;       "This method is the Gradient-expansion algorithm which
;       combines the best features of the gradient search with
;       the method of linearizing the fitting function."
;
;       Iterations are performed until the chi square changes by
;       only TOL or until ITMAX iterations have been performed.
;
;       The initial guess of the parameter values should be
;       as close to the actual values as possible or the solution
;       may not converge.
;
;
; MODIFICATION HISTORY:
;       Written, DMS, RSI, September, 1982.
;       Does not iterate IF the first guess is good.  DMS, Oct, 1990.
;       Added CALL_PROCEDURE to make the function's name a parameter.
;              (Nov 1990)
;       12/14/92 - modified to reflect the changes in the 1991
;            edition of Bevington (eq. II-27) (jiy-suggested by CreaSo)
;       Mark Rivers, U of Chicago, Feb. 12, 1995
;           - Added following keywords: ITMAX, ITER, TOL, CHI2, NODERIVATIVE
;             These make the routine much more generally useful.
;           - Removed Oct. 1990 modification so the routine does one iteration
;             even IF first guess is good. Required to get meaningful output
;             for errors.
;           - Added forward difference derivative calculations required for
;             NODERIVATIVE keyword.
;           - Fixed a bug: PDER was passed to user's procedure on first call,
;             but was not defined. Thus, user's procedure might not calculate
;             it, but the result was THEN used.
;
;      Steve Penton, RSI, June 1996.
;            - Changed SIGMAA to SIGMA to be consistant with other fitting
;              routines.
;            - Changed CHI2 to CHISQ to be consistant with other fitting
;              routines.
;            - Changed W to Weights to be consistant with other fitting
;              routines.
;            _ Updated docs regarding weighing.
;
;      Chris Torrence, RSI, Jan,June 2000.
;         - Fixed bug: if A only had 1 term, it was passed to user procedure
;           as an array. Now ensure it is a scalar.
;         - Added more info to error messages.
;         - Added /DOUBLE keyword.
;
;      MKC, 31/01/04
;         - adapted for use in fortran fro IDL CURVEFIT procedure
;         - Removed arguments and keywords : sigma, itmax, iter, tol, chi2, noderivative, chisq, double, fucntion_name
;
;-
FUNCTION RCS_CURVEFIT, x, y, weights, a

       ;Name of function to fit

       function_name = "ies_sgausscv"


	   tol = 1d-3  ;Convergence tol
	   itmax = 20     ;Maximum # iterations

       nterms = 3         ; # of parameters
       nY = n_elements(y)
       nfree = nY - nterms ; Degrees of freedom

       IF nfree LE 0 THEN MESSAGE, $
		'Number of parameters in A must be less than number of dependent values in Y.'

       IF (nterms EQ 1) THEN a = a[0]   ; Ensure a is a scalar
       flambda = 0.001d                   ;Initial lambda
       diag = lindgen(nterms)*(nterms+1) ; Subscripts of diagonal elements

;      Define the partial derivative array

       pder = fltarr(n_elements(y), nterms)

	error_msg1 = 'Result F from "'+ $
		Function_name+'" must have same number of elements as Y.'
	error_msg2 = 'PDER from "'+ $
		Function_name+'"  must be of size N_ELEMENTS(Y) by N_ELEMENTS(A).'

       FOR iter = 1, itmax DO BEGIN      ; Iteration loop

;         Evaluate alpha and beta matricies.

             ; The user's procedure will return partial derivatives
             call_procedure, function_name, x, a, yfit, pder
			IF (N_ELEMENTS(yfit) NE nY) THEN MESSAGE, error_msg1

          IF nterms EQ 1 THEN pder = reform(pder, n_elements(y), 1)
			IF (NOT ARRAY_EQUAL(SIZE(pder,/DIM),[nY,nterms])) THEN $
				MESSAGE, error_msg2

          beta = rcs_multiply((y-yfit)*Weights, pder, 1, nY, 3)

          alpha = rcs_multiply(transpose(pder), pder*rcs_multiply(Weights, (fltarr(nterms)+1),nY,1,3), 3,nY,3)

          ; save current values of return parameters

          sigma1 = sqrt( 1.0 / alpha[diag] )           ; Current sigma.
          sigma  = sigma1

          chisq1 = total(Weights*(y-yfit)^2)/nfree     ; Current chi squared.
          chisq = chisq1

          yfit1 = yfit

          done_early = chisq1 LT total(abs(y))/1d7/nfree
          IF done_early THEN GOTO, done

          c = sqrt(alpha[diag])

          c = rcs_multiply(c,c,3,1,3)

          lambdaCount = 0

          REPEAT BEGIN

             lambdaCount = lambdaCount + 1

             ; Normalize alpha to have unit diagonal.

             array = alpha / c

             ; Augment the diagonal.

             array[diag] = array[diag]*(1.+flambda)

             ; Invert modified curvature matrix to find new parameters.

             IF n_elements(array) EQ 1 THEN array = (1.0 / array) $
             ELSE array = rcs_invert(array)

             b = a + rcs_multiply(array/c,transpose(beta),3,3,1)          ; New params
             IF (nterms EQ 1) THEN b = b[0]             ; Ensure b is a scalar

             call_procedure, function_name, x, b, yfit  ; Evaluate function
             chisq = total(Weights*(y-yfit)^2)/nfree    ; New chisq
             sigma = sqrt(array[diag]/alpha[diag])      ; New sigma

             IF (finite(chisq) EQ 0) OR $
                  (lambdaCount GT 30 AND chisq GE chisq1) THEN BEGIN

                ; Reject changes made this iteration, use old values.

                yfit  = yfit1
                sigma = sigma1
                chisq = chisq1

                MESSAGE,'Failed to converge- CHISQ increasing without bound.', $
                   /INFORMATIONAL

                GOTO, done

             ENDIF

             flambda = flambda*10.               ; Assume fit got worse

          ENDREP UNTIL chisq LE chisq1

          flambda = flambda/100.

          a=b                                    ; Save new parameter estimate.

          IF ((chisq1-chisq)/chisq1) LE tol THEN GOTO,done   ;Finished?
       ENDFOR                        ;iteration loop

       iterationStr = STRTRIM(itmax,2)+' iteration' + (['','s'])[itmax NE 1]
       MESSAGE, 'Failed to converge after '+iterationStr+'.', $
          /INFORMATIONAL

done:  IF done_early THEN iter = iter - 1
       return,yfit          ; return result
END
