;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_DETECTOR_CLEAN
;;
; Purpose     :	Applies corrections to single IES detector sample.
;
; Explanation :	Corrects for pedestal shift.
;
; Use         : < rcs_detector_clean, npeds, pedestal, array, new_starts, old_starts, $
;                                     new_bands, nbands, minimum, FLAGS=flags
;
; Inputs      : npeds            : INT no of pedestal channels
;               pedestal         : FLOAT pedestal position in ADC bins
;               array            : FLTARR(nbands) containing input data for each energy channel
;               new_starts       : INTARR(new_bands) containing new energy channel start bin positions
;               old_starts       : INTARR(nbands) containing old energy channel start bin positions
;               new_bands        : INT new no of energy channels
;               nbands           : INT old no of energy channels
;               minimum          : FLOAT smallest non-zero data sample.
;
; Opt. Inputs : None.
;
; Outputs     : yy : FLTARR[new_bands]
;
; Opt. Outputs:	None.
;
; Keywords    : FLAGS    : LONG data quality flag
;
; Written     :	Version 0.0, 17/12/03
;                 Adapted from rcs do clean sample.
;
; Modified    : Version 0.1, 26/01/04
;                 Moved rcs do cross correlation and changed argument list.
;
; Version     :	Version 0.1, 26/01/04
;-
;**********************************************************

FUNCTION rcs_detector_clean, npeds, pedestal, array, $
    new_starts, old_starts, new_bands, nbands, minimum, FLAGS=flags

  ; rebin y data

  ; IES_INTERP

  ; form blank array

    yy = FLTARR(new_bands)

    ; get channel data

    yc = array[npeds:*]

    ; get cumulative counts from last channel
    ; omit pedestal channels from interpolation

    FOR k = npeds+1, nbands-1 DO yc(nbands-1-k) = yc(nbands-1-k) + yc(nbands-k)

    ; take log (may have fractional counts per spin)
    ; summed channel should not be present
    ; set any zero samples to 0.1*minimum

    yc = ALOG(yc > (0.1*minimum))

    ; get logs of channel boundaries, ensure any invalid LUT 0's do not cause a problem.

    xtemp = ALOG(old_starts[npeds:*] > 0.1)

    ; get LOG bin positions for each sample corrected for pedestal shift
    ; remove any zeroes

    typ = ALOG((new_starts[0:new_bands-1] + pedestal) > 0.1)

    ; set warning flag if new bins outside of valid data range
    ; NB xtemp(nbands-1) is the start boundary of the last channel

    IF typ[0] LT xtemp[0] THEN BEGIN
      IF N_ELEMENTS(flags) NE 0 THEN flags = flags AND '00000100'XL
    ENDIF

    IF typ[new_bands-1] GT xtemp[nbands-npeds-1] THEN BEGIN
      IF N_ELEMENTS(flags) NE 0 THEN flags = flags AND '00000200'XL
    ENDIF

    ; interpolate data using linear form in LOG(e) vs LOG(x)

    IF N_ELEMENTS(xtemp) GT 1 THEN $
      yy = EXP ( ies_interpol ( yc, xtemp, typ ) ) $
    ELSE $
      yy = EXP(yc(0))

    ; get counts rather than cumulative counts (except last channel)

    FOR k = 0, new_bands-2 DO yy[k] = yy[k] - yy[k+1]

    ; set any resampled value less than 0.5*minimum to zero

    list = WHERE ( yy LT (0.5*minimum), count)

    IF count GT 0 THEN yy[list] = 0

    RETURN, yy

END
