;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_DO_CLEAN_SAMPLE
;
; Purpose     :	Applies corrections to single IES sample.
;
; Explanation :	Corrects for pedestal shift.
;               It uses a COMMON block to store information needed for each call.
;
; Use         : < rcs_do_clean_sample, sc, time, lut_no, array, ch_positions, BM=bm >
;
; Inputs      : sc           : STRING character giving spacecraft ID
;               time         : DOUBLE TAI time of EDB
;               lut          : INT LUT no
;               array        : FLTARR(nsectors,ndetectors,nbands) counts
;               ch_positions : INTARR(2,nbands,ndetectors,nluts) old energy channel positions in bins for each detector and LUT
;               new_ch_positions : INTARR(2,newbands,ndetectors,nluts) new energy channel positions in bins for each detector and LUT
;
; Opt. Inputs : None.
;
; Outputs     : array        : FLTARR(nsectors,ndetectors,newbands) counts
;               ch_positions : INTARR(2,newbands,ndetectors,nluts) new energy channel positions in bins for each detector and LUT
;
; Opt. Outputs:	None.
;
; Keywords    : BM        : FLAG indicates BM data
;               PEDESTALS : FLTARR(5,*) pedestal info array
;               FLAGS     : LONG data quality flag
;
; Written     :	Version 0.0, 17/12/03
;                 Adapted from rcs do clean.
;
; Modified    : Version 0.1, 26/01/04
;                 Moved rcs do cross correlation from rcs_detector_clean.
;
; Version     :	Version 0.1, 26/01/04
;-
;**********************************************************

PRO rcs_do_clean_sample, sc, time, lut, array, ch_positions, new_ch_positions, BM=bm, PEDESTALS=pedestals, FLAGS=flags

  ; RCS_DO_CLEAN

  no_of_detectors = (SIZE(ch_positions))[3]

  nbands = (SIZE(ch_positions))[2]

  new_bands = (SIZE(new_ch_positions))[2]

  IF KEYWORD_SET(bm) THEN BEGIN

    npeds = 4

    no_of_sectors = 16

  ENDIF ELSE BEGIN

    npeds = 2

    no_of_sectors = 1

  ENDELSE

  ; array        : FLTARR(nsectors,ndetectors,nbands)

  array = REFORM(array, no_of_sectors, no_of_detectors, nbands)

  ; get pedestal info
  ; Inputs      : npeds         : INT no. of pedestal channels
  ;               array         : FLTARR( nsectors, ndetectors, nbands ) containing input data
  ;               times         : DBLARR(npoints) TAI times of samples
  ;               nsectors      : INT no. of sectors.
  ; Outputs     : pedestals : FLTARR ( 4, ndetectors, nsectors)
  ;                           0 : pedestal shift from central bin for cross_correlation algorithm
  ;                           1 : pedestal shift from central bin for centroid algorithm
  ;                           2 : occupancy defined as the ratio of non-pedestal counts to measured total counts
  ;                           3 : visible counts defined as ratio of counts in data channels vs expected total counts

  ; RCS_PEDESTALS

  ; set up pedestal info output

  pedestals = FLTARR(5, no_of_detectors, no_of_sectors)

  ; get expected total no. of counts
  ; these nos are derived from histogram calibration data

  tot_exp_counts = ([78230L, 73770L, 62136L, 39970L])[lut mod 4] / FLOAT(no_of_sectors)

  ; get data min ignoring zeros
  ; can ignore flags since these will not decrease data if set to 255 and are zeroed otherwise

  non_zero_list = WHERE (array GT 0, count)

  IF count EQ 0 THEN minimum = 1 ELSE minimum = MIN(array[non_zero_list])

  ; last channel is integral channel so does not need end boundary

  new_yp = FLOAT(REFORM(new_ch_positions[0,*,0,0]))

  ; get array for new output data

  new_output = FLTARR ( no_of_sectors, no_of_detectors, new_bands)

  ; loop over each detector

  FOR d = 0, no_of_detectors-1 DO BEGIN

    ; get calibration info samples for cross_correlation
    ; qsamples[npeds,nshifts] is an array containing shifted pedestal calibration data sampled as for the real data
    ; only the first npeds energy channels are kept
    ; do not search for calibration data at specific temperature
    ; this is used to restrict the search for max cross-correlation
    ; to samples with the same peak position as the data

    rcs_pedestal_samples, sc, d, lut, samples, max_sample_lists, pedestal_centre, channel_centres, pedestal_posn, BM=bm

    ; NB last channel is integral channel so does not need bin boundary

    ; get channel start boundaries for all luts for all channels

    channel_starts = REFORM(ch_positions[0,*,d,lut])

    ; loop over each sample sectors getting pedestal info

    FOR s = 0, no_of_sectors-1 DO BEGIN

      p = rcs_do_cross_correlation(npeds, array, samples, tot_exp_counts)

      pedestals[0,d,s] = pedestal_posn - (p-50)/10.0

      new_output[s,d,*]  = rcs_detector_clean ( npeds, pedestals[0,d,s], REFORM(array[s,d,*]), samples, $
                                                new_yp, channel_starts, new_bands, nbands, minimum, $
                                                FLAGS=flags )

    ENDFOR ; loop over s

  ENDFOR ; loop over d

  ; set up outputs

  array = new_output

END
