;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_DO_CROSS_CORRELATION
;
; Purpose     : Locates pedestal bin position and optionally subtracts pedestal from each sample.
;
; Explanation :	This routine calculates pedestal shift to a fraction of a channel by cross-correlating the known
;               pedestal profile with the sampled data.
;               The pedestal contribution may then be subtracted if required.
;               Uses all the pedestal channels and normalizes the shifted histogram data to the expected number
;               of pedestal counts.
;
; Use         : < pedestals = rcs_do_cross_correlation(npeds, array, samples, tot_exp_counts) >
;
; Inputs      : npeds            : INT no of pedestal channels
;               array            : FLTARR(nbands) containing input data for each energy channel
;               samples          : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
;                                  data sampled as for the real data. Only the first npeds energy channels are kept.
;                                  Whole spectrum is normalized to one.
;               tot_exp_counts   : LONG total no. of counts expected for integration time (if no compression).
;
; Opt. Inputs : None.
;
; Outputs     : pedindex : FLOAT
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 17/09/00
;
; Modified    :	Version 0.1, 9/2/01
;                 Converted occupancy and missing counts to percentage and out of 10 respectively.
;               Version 0.2, 20/07/01
;                 Used ies_runs.
;               Version 0.3, 14/05/02
;                 Used rcs_check_ies to get expected counts.
;               Version 0.4, 24/10/03
;                 Changed to lut_list argument.
;               Version 0.5, 20/11/03
;                 Separated calibration info -- removed keywords CH_POSITIONS, COUNT_CAL, ENERGY_CAL, ENERGY_EFF
;                 Used rcs pedestal samples.
;                 Added BM keyword.
;                 Removed ped_chans.
;                 Separated from rcs pedestals.
;                 Incorporated cep pedestals.
;                 Corrected bug in cep pedestals if ERRORS flag not used.
;                 Change ERRORS keyword to CHISQ.
;                 Used ies get normalization.
;                 Used cep pedestal info and rcs pedestal info.
;               Version 0.6, 18/12/03
;                 Constructed rcs do cross correlation from ies do cross correlation.
;                 Removed max sample lists and KEYWORDS.
;
; Version     :	Version 0.6, 18/12/03
;-
;**********************************************************

FUNCTION rcs_do_cross_correlation, npeds, array, samples, tot_exp_counts

  ; get pedestal position from cross-correlation

  ; ped_array is pedestal channels for given sector and given sample [npeds]

  ped_array = array[0:npeds-1]

  ; find pedestal channel maximum

  max_ped_array = MAX(ped_array,max_pos)

  ; samples  : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
  ;            data sampled as for the real data.
  ;            These have been normalized to unity for the whole spectrum.

  ; tot_ped_array is the sum of counts in ped_array

  tot_ped_array = TOTAL(ped_array)

  ; form a 2-D array replicating ped_array for each shifted sample with same peak
  ; ped2d_array  : FLTARR[npeds,nshifts] pedestal data replicated

  ped2d_array = ped_array#REPLICATE(1,(SIZE(samples))[2])

  ; normalize samples

  ; get total no of counts in pedestal and signal channels

  tot_array = TOTAL(array)

  ; normalize ssamples to expected no of counts taking into account signal counts
  ; reduce counts in predicted pedestal samples by observed signal counts
  ; NB do not have actual total counts

  normalization = tot_exp_counts-tot_array+tot_ped_array

  ; NB ssamples : FLTARR[npeds,nshifts], normalization : FLOAT

  ssamples = samples*normalization

  ; take residuals with sampled pedestal array

  residuals =  ped2d_array - ssamples

  ; minimize sum of chisq over the pedestal channels where chisq is the square of difference between the
  ; actual counts and the predicted counts for each shift divided by the actual counts

  smsqs  = TOTAL((residuals*residuals)/(ped2d_array>1),1)  ; takes into account Poisson errors

  ; find min

  dummy = MIN ( smsqs, p )

  ; return min index

  RETURN, p

END