;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_INVERT
;
; Purpose     : Gets pedestal sample info for given detector and LUT.
;
; Explanation :	Gets pedestal sample info for given detector and LUT.
;
; Use         : < rcs_ped_samples, sc, detector, lut, samples, max_samples_lists, pedestal_centre, channel_centres, $
;                                       pedestal_posn, BM=bm >
;
; Inputs      : sc       : STRING character SC
;               detector : INT detector no. 0-8
;               lut      : INT LUT no
;
; Opt. Inputs : None.
;
; Outputs     : samples          : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
;                                  data sampled as for the real data. Only the first npeds energy channels are kept.
;               max_sample_list  : FLTARR(1+nshifts, npeds) indices of shifts with peak in channel 1+channel.
;                                  Element 0 contains no of indices.max_samples contains the index of the peak
;                                  for each shift. This is used to restrict the search for max cross-correlation
;                                  to samples with the same peak position as the data
;               pedestal_centre  : FLOAT position of center of pedestal channels.
;               channel_centres  : FLTARR(npeds) position of centers of pedestal channels.
;               pedestal_posn    : FLOAT result of fit to pedestal.
;               pedestal_width   : FLOAT result of fit to pedestal (Gaussian width).
;
; Opt. Outputs:	None.
;
; Keywords    : BM           : FLAG indicating BM rather than NM.
;               FORCE        : FLAG indicating should force set up of calibration data and
;                                   calculation
;
; Written     :	Version 0.0, 21/11/03
;                 Separated from rcs pedestals.
;
; Modified    : Version 0.1, 12/12/03
;                 Added sc argument.
;               Version 0.2, 19/01/04
;                 Corrected bug : COMMON block not reset for new SC.
;               Version 0.4, 02/02/04
;                 Added FORCE keyword.
;                 Added pedestal_width to argument list
;
; Version     :	Version 0.4, 02/02/04
;-
;**********************************************************

FUNCTION rcs_invert, m

  inv = FLTARR(3,3)

  inv[0,0] = m[1,1]*m[2,2]-m[2,1]*m[1,2]

  inv[0,1] = - ( m[0,1]*m[2,2] - m[2,1]*m[0,2] )

  inv[0,2] = m[0,1]*m[1,2] - m[1,1]*m[0,2]

  inv[1,0] = - ( m[1,0]*m[2,2] - m[2,0]*m[1,2] )

  inv[1,1] = m[0,0]*m[2,2] - m[2,0]*m[0,2]

  inv[1,2] = - ( m[0,0]*m[1,2] - m[1,0]*m[0,2] )

  inv[2,0] = m[1,0]*m[2,1] - m[2,0]*m[1,1]

  inv[2,1] = - ( m[0,0]*m[2,1] - m[2,0]*m[0,1] )

  inv[2,2] = m[0,0]*m[1,1] - m[1,0]*m[0,1]

  determinant = m[0,0]*inv[0,0] + m[0,1]*inv[1,0] + m[0,2]*inv[2,0]

  IF determinant EQ 0.0 THEN RETURN, inv

  RETURN, inv/determinant

END